import logging
import types
from functools import partial
from typing import Any, Dict, List

from detectors.methods.templates import DetectorWrapper

from .dice import Dice
from .doctor import doctor
from .energy import energy
from .entropy import entropy
from .gradnorm import gradnorm
from .igeood_logits import IgeoodLogits
from .kl_matching import KLMatching
from .knn_euclides import KnnEuclides
from .mahalanobis import Mahalanobis
from .maxcosine import MaxCosineSimilarity
from .msp import msp
from .odin import odin
from .projection import Projection
from .react import ReAct
from .relative_mahalanobis import RelativeMahalanobis
from .vim import ViM

_logger = logging.getLogger(__name__)

detectors_registry = {
    # hyperparameter free detectors
    "msp": msp,
    "kl_matching": KLMatching,
    "vim": ViM,
    "maxcosine": MaxCosineSimilarity,
    "entropy": entropy,
    # hyperparameter detectors
    "odin": odin,
    "doctor": doctor,
    "energy": energy,
    "dice": Dice,
    "react": ReAct,
    "igeood_logits": IgeoodLogits,
    "gradnorm": gradnorm,
    "knn_euclides": KnnEuclides,
    # Features based detectors
    "mahalanobis": Mahalanobis,
    "relative_mahalanobis": RelativeMahalanobis,
    "projection": Projection,
}


def register_detector(name: str):
    def decorator(f):
        detectors_registry[name] = f
        return f

    return decorator


def create_detector(detector_name: str, **kwargs) -> DetectorWrapper:
    model = kwargs.pop("model", None)
    if detector_name not in detectors_registry:
        raise ValueError(f"Unknown detector: {detector_name}")
    if not isinstance(detectors_registry[detector_name], types.FunctionType):
        return DetectorWrapper(detectors_registry[detector_name](model=model, **kwargs), **kwargs)
    return DetectorWrapper(partial(detectors_registry[detector_name], model=model, **kwargs), **kwargs)


def list_detectors() -> List[str]:
    return list(k for k in detectors_registry.keys() if detectors_registry[k] is not None)


def create_hyperparameters(detector_name: str) -> Dict[str, Any]:
    import importlib

    try:
        module = importlib.import_module(f"src.methods.{detector_name}")
        hyperparameters = module.HYPERPARAMETERS
    except ModuleNotFoundError:
        raise ValueError(f"Unknown detector: {detector_name}")
    except AttributeError:
        hyperparameters = {}
    return hyperparameters
