import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import glob
import os
import re


def compute_score_ratio(folder_path, player_self, player_opp, output_path="score_ratio_stats.txt"):
    strategy_stats = {}

    for csv_file in glob.glob(os.path.join(folder_path, "*.csv")):
        filename = os.path.basename(csv_file)
        filename_no_ext = filename.replace('.csv', '')
        segments = filename_no_ext.split('_')
        if len(segments) < 3:
            continue
        method_name, file_playerA, file_playerB = segments[0], segments[1], segments[3]

        if file_playerA != player_self or file_playerB != player_opp:
            continue

        df = pd.read_csv(csv_file)
        if 'roundscore_A' not in df.columns or 'roundscore_B' not in df.columns or 'match' not in df.columns:
            continue
        df = df.sort_values(by='match')
        match_ids = df['match'].unique()
        self_match_ratios = []

        for match_id in match_ids:
            match_df = df[df['match'] == match_id]
            last_row = match_df.iloc[-1]
            scoreA = last_row['roundscore_A']
            scoreB = last_row['roundscore_B']
            total = scoreA + scoreB
            if total == 0:
                continue
            self_ratio = scoreA / total 
            self_match_ratios.append(self_ratio)

        if self_match_ratios:
            avg = np.mean(self_match_ratios)
            std = np.std(self_match_ratios)
            strategy_stats[method_name] = (avg, std)

    name_map = {
        'bc': 'BC',
        'dd': 'DD',
        'dbc': 'DBC',
        'dp': 'DP',
        'ddgi': 'DDGI',
    }

    strategy_stats = {
        name_map.get(k, k): v for k, v in strategy_stats.items() if k in name_map
    }

    # Write to text file
    with open(output_path, 'a') as f:
        f.write(f'Score Ratio Statistics: {player_self} vs {player_opp}\n')
        f.write('----------------------------------------\n')
        for method, (avg, std) in strategy_stats.items():
            f.write(f'{method}: {avg:.4f} ± {std:.4f}\n')
        f.write('----------------------------------------\n\n')


if __name__ == "__main__":
    player_list = ['Viktor AXELSEN', 'Kento MOMOTA', 'CHOU Tien Chen']
    for p_self in player_list:
        for p_opp in player_list:
            if p_self != p_opp:
                compute_score_ratio(
                    folder_path="./evaluation/data/badminton",  # folder
                    player_self=p_self,
                    player_opp=p_opp,
                    output_path=f"./evaluation/plot/badminton/score_ratio.txt"
                )