import numpy as np
from sklearn.metrics import confusion_matrix
import math

def evaluate(actual, predicted):
    """
    Computes accuracy, sensitivity (recall), specificity, precision, F-measure, and G-mean.
    Works for both binary and multi-class classification.
    """
    actual = np.asarray(actual).flatten()
    predicted = np.asarray(predicted).flatten()

    # Ensure confusion matrix has correct shape regardless of label presence
    nclass = max(np.max(actual), np.max(predicted)) + 1
    cm = confusion_matrix(actual, predicted, labels=list(range(nclass)))

    if nclass == 2:
        tn, fp, fn, tp = cm.ravel()
        p = tp + fn
        n = tn + fp
        N = p + n

        sensitivity = 100 * tp / p if p else 0
        specificity = 100 * tn / n if n else 0
        accuracy = 100 * (tp + tn) / N if N else 0
        precision = 100 * tp / (tp + fp) if (tp + fp) else 0
        f_measure = 2 * (precision * sensitivity) / (precision + sensitivity) if (precision + sensitivity) else 0
        gmean = 100 * math.sqrt((tp / p) * (tn / n)) if p and n else 0

    else:
        # Multi-class: macro average
        diag = np.diag(cm)
        row_sum = cm.sum(axis=1)
        col_sum = cm.sum(axis=0)
        p = np.sum(row_sum)
        n = p

        sensitivity = 100 * np.mean(diag / row_sum.clip(min=1))
        specificity = 100 * np.mean((col_sum - diag) / col_sum.clip(min=1))
        accuracy = 100 * np.sum(diag) / p if p else 0
        precision = 100 * np.mean(diag / col_sum.clip(min=1))
        f_measure = 2 * (precision * sensitivity) / (precision + sensitivity) if (precision + sensitivity) else 0
        gmean = 100 * math.sqrt(sensitivity / 100 * specificity / 100)

    return [accuracy, sensitivity, specificity, precision, f_measure, gmean]
