function W_reg = regulate_weights_using_StaR(W_init, rho_min, rho_target)
%REGULATE_WEIGHTS Regulates the spectral values of a weight matrix.
%
%   W_reg = REGULATE_WEIGHTS(W_init, rho_min, rho_target) returns a matrix
%   W_reg that has the same directional structure as W_init, but with its 
%   singular values rescaled to lie within [rho_min, rho_target].
%
%   Inputs:
%       - W_init:        Initial random weight matrix (e.g., randn(d, h))
%       - rho_min:       Minimum desired singular value (>= 0)
%       - rho_target:    Maximum desired singular value (spectral radius)
%
%   Output:
%       - W_reg:         Spectrally regulated weight matrix

    % --------------------------------------------------------
    % Step 1: SVD of input weight matrix
    % --------------------------------------------------------
    [U, S, V] = svd(W_init, 'econ');
    singular_vals = diag(S); % Extract singular values
    
    % --------------------------------------------------------
    % Step 2: Normalize to [0, 1] and rescale to [rho_min, rho_target]
    % --------------------------------------------------------
    s_min = min(singular_vals);
    s_max = max(singular_vals);
    
    % Avoid division by zero in case s_min == s_max
    singular_vals_norm = (singular_vals - s_min) / (s_max - s_min + eps);
    
    % Rescale to target range
    singular_vals_scaled = singular_vals_norm * (rho_target - rho_min) + rho_min;
    
    % --------------------------------------------------------
    % Step 3: Reconstruct weight matrix with regulated spectrum
    % --------------------------------------------------------
    S_scaled = diag(singular_vals_scaled);
    W_reg = U * S_scaled * V';
end
