import numpy as np
from scipy.linalg import pinv

def l2_weights(A, b, C):
    """
    Solves (A^T A + I/C)^-1 A^T b if features < samples
    or A^T (A A^T + I/C)^-1 b otherwise.
    
    Args:
        A: [n_samples x n_features] design matrix
        b: [n_samples x n_outputs] target matrix
        C: regularization parameter (non-zero positive scalar)
        
    Returns:
        x: [n_features x n_outputs] weight matrix
    """
    n_samples, n_features = A.shape
    if C == 0:
        return pinv(A) @ b
    if n_features < n_samples:
        return np.linalg.solve((np.eye(n_features) / C + A.T @ A), A.T @ b)
    else:
        return A.T @ np.linalg.solve((np.eye(n_samples) / C + A @ A.T), b)
