function W_reg = regulate_weights_using_StaR(W_init, rho_min, rho_target)
%REGULATE_WEIGHTS_USING_STAR Regulates the singular values of a weight matrix.
%
%   W_reg = regulate_weights_using_StaR(W_init, rho_min, rho_target)
%   returns a weight matrix W_reg with singular values scaled to lie
%   within [rho_min, rho_target], preserving directional structure.
%
%   Inputs:
%       - W_init:      Initial random weight matrix (e.g., randn(d, h))
%       - rho_min:     Minimum desired singular value (>= 0)
%       - rho_target:  Maximum desired spectral norm (rho_target > rho_min)
%
%   Output:
%       - W_reg:       Spectrally regulated weight matrix

    % Step 1: Perform SVD
    [U, S, V] = svd(W_init, 'econ');
    singular_vals = diag(S);

    % Step 2: Normalize singular values to [0, 1]
    s_min = min(singular_vals);
    s_max = max(singular_vals);
    
    % Avoid division by zero if all singular values are equal
    singular_vals_norm = (singular_vals - s_min) / (s_max - s_min + eps);

    % Step 3: Rescale to [rho_min, rho_target]
    singular_vals_scaled = singular_vals_norm * (rho_target - rho_min) + rho_min;

    % Step 4: Reconstruct regulated weight matrix
    S_scaled = diag(singular_vals_scaled);
    W_reg = U * S_scaled * V';
end
