# Import Python packages.
from typing import Any, Dict, List, Mapping, Sequence, Tuple

# Import developing packages.
import fin_tech_py_toolkit as lib


def prepare(
    name: str, prefix: str, suffix: str, /, *, label: str = ""  # noqa: W504
) -> Tuple[lib.transforms.BaseTransformPandas, Sequence[Any], Mapping[str, Any]]:
    r"""
    Prepare transformation.

    Args
    ----
    - name
        Transformation name.
    - prefix
        Cache prefix.
    - suffix
        Cache suffix.

    Returns
    -------
    - transform
        A transformation.
    - fit_args
        Positional arguments for parameter fitting.
    - fit_kwargs
        Keyword arguments for parameter fitting.
    """
    # Prepare preprecoessing.
    transform: lib.transforms.BaseTransformPandas
    fit_args: List[Any]
    fit_kwargs: Dict[str, Any]
    if name == "cca":
        # CCA encoder.
        transform = lib.transforms.TransformCCAPandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(nan=0, ddof=1, ood="<unk>", normalize_degree="log")
    elif name == "cca-degree":
        # CCA encoder.
        transform = lib.transforms.TransformCCAPandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(
            nan=0, ddof=1, ood="<unk>", normalize_degree="log", aggregates=["deg", "null"]
        )
    elif name == "cca-distribute":
        # CCA encoder.
        transform = lib.transforms.TransformCCAPandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(
            nan=0, ddof=1, ood="<unk>", normalize_degree="log", aggregates=["mean", "std"]
        )
    elif name == "cca-range":
        # CCA encoder.
        transform = lib.transforms.TransformCCAPandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(
            nan=0, ddof=1, ood="<unk>", normalize_degree="log", aggregates=["min", "max"]
        )
    elif name == "count":
        # Count encoder.
        transform = lib.transforms.TransformCountizePandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(unk=0, ood="<unk>", avoid_collide=False)
    elif name == "sdv":
        # Count encoder.
        transform = lib.transforms.TransformCountizePandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(unk=0, ood="<unk>", avoid_collide=False)
    elif name == "catboost":
        # CatBoost encoder.
        transform = lib.transforms.TransformCatBoostEncodePandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(
            category_encoders_init_kwargs=dict(handle_unknown="value", random_state=42),
            target=("categorical", label),
        )
    elif name == "discard":
        # Discard categorical data.
        transform = lib.transforms.TransformDiscardEncodePandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = {}
    else:
        # Unsupporting transformation.
        raise NotImplementedError
    return transform, fit_args, fit_kwargs
