# Import Python packages.
from typing import Mapping

# Import external packages.
import numpy as np
from sklearn.metrics import (  # type: ignore[import-untyped]
    accuracy_score,
    average_precision_score,
    f1_score,
    precision_score,
    recall_score,
    roc_auc_score,
)

# Import developing library.
from fin_tech_py_toolkit.types import NPFLOATS, NPINTS


def evaluate(scores_: NPFLOATS, labels: NPINTS, /) -> Mapping[str, float]:
    r"""
    Evaluate binary classification.

    Args
    ----
    - scores
        Output scores.
    - labels
        Target labels.

    Returns
    -------
    - profile
        Evaluation profile.
    """
    # Get output labels.
    labels_ = (scores_ >= 0.5).astype(int)

    # Safety check.
    assert np.logical_and(scores_ >= 0, scores_ <= 1).all().item()
    assert np.logical_and(labels_ >= 0, labels_ <= 1).all().item()
    assert np.logical_and(labels >= 0, labels <= 1).all().item()

    # Compute all metrics.
    rocauc = roc_auc_score(labels, scores_)
    ap = average_precision_score(labels, scores_)
    f1 = f1_score(labels, labels_, zero_division=0.0)
    acc = accuracy_score(labels, labels_)
    prec = precision_score(labels, labels_, zero_division=0.0)
    recl = recall_score(labels, labels_, zero_division=0.0)
    return {"rocauc": rocauc, "ap": ap, "f1": f1, "acc": acc, "prec": prec, "recl": recl}
