# Import Python packages.
from typing import Any, Dict, List, Mapping, Optional, Sequence, Tuple, TypeVar

# Import external packages.
import pandas as pd

# Import relatively from other modules.
from ...data import DataTabular
from ...types import NPANYS
from ..base import BaseDataset


# Self types.
SelfDatasetTabular = TypeVar("SelfDatasetTabular", bound="DatasetTabular")


class DatasetTabular(BaseDataset[DataTabular]):
    r"""
    Dataset of tabular data.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "dataset.tabular._base"

    def memalias(
        self: SelfDatasetTabular,
        memory: List[DataTabular],
        memory_names: Sequence[str],
        /,
        *args: Any,
        addresses: Mapping[str, str] = {},
        sorts: Optional[Tuple[str, str]] = None,
        **kwargs: Any,
    ) -> SelfDatasetTabular:
        r"""
        Alias data memory by the memory of the class.

        Args
        ----
        - memory
            Data memory.
        - memory_names
            Named indices of memory slots.
        - addresses
            Named addresses corresponding to memory slots respectively.
        - sorts
            Column and row sorting algorithms for disambiguiation on homogeneous tabular data from
            each named addresses.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Safety check.
        assert sorts is not None, (
            "For safety, silent tabular disambiguition is not allowed, but disambuigition sorting"
            " algorithms are not provided."
        )

        # Homogeneity criterion annotations.
        criterion: Any

        # Check homogeneity of data containers.
        criterion = tuple(next(iter(memory))._content.columns)
        invalid = [
            f'"{name:s}"'
            for name, data in zip(memory_names, memory)
            if tuple(data._content.columns) != criterion
        ]
        invalid_ = [*invalid[:3], "..."] if len(invalid) > 3 else invalid
        assert not invalid, "Find heterogeneous tabular columns at memory slots: {:s}".format(
            ", ".join(invalid_)
        )

        # Check homogeneity of data ordering.
        criterion = sorts
        invalid = [
            f'"{name:s}"'
            for name, data in zip(memory_names, memory)
            if (data.sort_columns, data.sort_rows) != criterion
        ]
        invalid_ = [*invalid[:3], "..."] if len(invalid) > 3 else invalid
        assert (
            not invalid
        ), "Find heterogeneous tabular disambiguition sorting at memory slots: {:s}".format(
            ", ".join(invalid_)
        )

        # Super call.
        BaseDataset.memalias(self, memory, memory_names, *args, addresses=addresses, **kwargs)

        # Save essential attributes.
        self._sorts = sorts
        return self

    def get_numeric_data(self: SelfDatasetTabular, /) -> Mapping[str, NPANYS]:
        r"""
        Get numeric data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the transformation.
        """
        # For memory linking dataset, never save numeric data to avoid duplication.
        data: Dict[str, NPANYS]
        if self.link:
            # Return empty numeric data directly.
            data = {}
        else:
            # Collect all tabular data in memory as numeric arraies.
            data = {}
            for prefix in self._addresses.keys():
                # Merge numeric arraies loaded from each address together.
                data.update(
                    {
                        f"{prefix:s}.{suffix:s}" if suffix else prefix: array
                        for suffix, array in (
                            self.memory[self._memory_indices[prefix]].to_numeric().items()
                        )
                    }
                )
        return data

    def set_numeric_data(
        self: SelfDatasetTabular, data: Mapping[str, NPANYS], /  # noqa: W504
    ) -> SelfDatasetTabular:
        r"""
        Set numeric data of the transformation.

        Args
        ----
        - data
            Numeric data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Group columns of the same tables together.
        # Since column information is not ready yet, use anonymous column titles.
        content: Dict[str, Dict[str, NPANYS]]
        content = {}
        for name, array in data.items():
            # Get table and column names.
            prefix, suffix = name.split(".")
            try:
                # Add column data to corresponding table.
                content[prefix][suffix] = array
            except KeyError:
                # If table does not exist, create a new buffer.
                content[prefix] = {suffix: array}
        self._content = {name: pd.DataFrame(datadict) for name, datadict in content.items()}
        return self
