# Import Python packages.
import warnings
from typing import Any, List, Mapping, Optional, TypeVar

# Import external packages.
import numpy as np
import pandas as pd

# Import relatively from other modules.
from ...data import DataTabular
from ...datasets import DatasetTabular, DatasetTabularSimple
from ...transforms import ErrorTransformUnsupportPartial
from .base import TransdatasetUnravel


# Type aliases.
Input = List[DatasetTabular]
Output = List[pd.DataFrame]


# Self types.
SelfTransdatasetUnravelTabular = TypeVar(
    "SelfTransdatasetUnravelTabular", bound="TransdatasetUnravelTabular"
)


class TransdatasetUnravelTabular(TransdatasetUnravel[DatasetTabular, pd.DataFrame]):
    r"""
    Transformation for unraveling tabular data into Pandas dataframes.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "unravel.dataset.tabular"

    def input(self: SelfTransdatasetUnravelTabular, raw: Any, /) -> Input:
        r"""
        Convert raw data into input to the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Nothing to be unraveled.
            return [
                DatasetTabularSimple.from_memalias(
                    [
                        DataTabular.from_numeric(
                            {"generic": np.array([], dtype=np.int64)},
                            sort_columns="alphabetic",
                            sort_rows="rankable",
                        )
                    ],
                    ["full"],
                    sorts=("alphabetic", "rankable"),
                )
            ]
        else:
            # All the other cases are not supported.
            raise ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into input domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def output(self: SelfTransdatasetUnravelTabular, raw: Any, /) -> Output:
        r"""
        Convert raw data into output from the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Nothing to be unraveled.
            return [pd.DataFrame({"generic": []})]
        else:
            # All the other cases are not supported.
            raise ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into output domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def transform(
        self: SelfTransdatasetUnravelTabular, input: Input, /, *args: Any, **kwargs: Any
    ) -> Output:
        r"""
        Transform input into output without inplacement.

        Args
        ----
        - input
            Input to the transformation.

        Returns
        -------
        - output
            Output from the transformation.
        """
        # Parse input memory with safety check.
        (dataset,) = input
        assert tuple(dataset.memory_names) == tuple(
            self.names
        ), "The unraveling dataset does not have expecting named slot indices."

        # Flatten all data container content into a collection of dataframes.
        memory: List[pd.DataFrame]
        memory = []
        for i, data in enumerate(dataset.memory):
            # Fetch dataframe content of each data container.
            sorts_ = (data.sort_columns, data.sort_rows)
            assert sorts_ == self.sorts, (
                f"The unraveling memory slot ({i:d}-th) does not use expecting"
                f' disambiguition sorting algorithms: (get "{str(sorts_):s}", expect'
                f' "{str(self.sorts):s}").'
            )
            memory.append(data._content if self.allow_alias else data._content.copy())
        return memory

    def inverse(
        self: SelfTransdatasetUnravelTabular,
        output: Output,
        /,
        *args: Any,
        sort_columns: Optional[str] = None,
        sort_rows: Optional[str] = None,
        **kwargs: Any,
    ) -> Input:
        r"""
        Inverse output into input without inplacement.

        Args
        ----
        - output
            Output from the transformation.
        - sort_columns
            Column sorting algorithm name in input domain.
            If not given, it will inherit from fitting parameters.
        - sort_rows
            Row sorting algorithm name in input domain.
            If not given, it will inherit from fitting parameters.

        Returns
        -------
        - input
            Input to the transformation.
        """
        # For some disambiguition sorting algorithms, we can not guarantee a perfect inverse, thus a
        # warning will be raised.
        sort_input_columns = self.sorts[0] if sort_columns is None else sort_columns
        sort_input_rows = self.sorts[1] if sort_rows is None else sort_rows
        if sort_input_columns == "identity" or sort_input_rows == "identity":
            # Identity sorting can be ambiguous
            warnings.warn(
                f'Input tabular domain uses "{str((sort_input_columns, sort_input_rows)):s}"'
                f" disambiguition, which may result in ambiguition in inverson.",
                RuntimeWarning,
            )

        # Safety check.
        assert len(output) == len(self.names), (
            f"Number of raveling dataframes ({len(output):d}) does not match expecting number of"
            f" dataframes ({len(self.names):d})."
        )

        # Merge dataframes into the same datasets with auto disambiguition.
        dataset = DatasetTabularSimple.from_memalias(
            [
                DataTabular.from_numeric(
                    {str(name): series.to_numpy() for name, series in dataframe.items()},
                    sort_columns=sort_input_columns,
                    sort_rows=sort_input_rows,
                )
                for dataframe in output
            ],
            self.names,
            sorts=(sort_input_columns, sort_input_rows),
        )
        return [dataset]

    def fit(
        self: SelfTransdatasetUnravelTabular,
        input: Input,
        output: Output,
        /,
        *args: Any,
        **kwargs: Any,
    ) -> SelfTransdatasetUnravelTabular:
        r"""
        Fit transformation parameters by example input and output.

        Args
        ----
        - input
            Example input to the transformation.
        - output
            Example output from the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Collect essential attributes of each dataset.
        (dataset,) = input
        self.sorts = dataset._sorts
        self.names = dataset.memory_names
        return self

    def get_alphabetic_data(self: SelfTransdatasetUnravelTabular, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # Collect essential attributes for operating datasets.
        return {"sorts": self.sorts, "names": self.names}

    def set_alphabetic_data(
        self: SelfTransdatasetUnravelTabular, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransdatasetUnravelTabular:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Safety check.
        assert "sorts" in data, (
            "Disambiguition sorting algorithms of data containers in the operating dataset are"
            " missing."
        )
        assert "names" in data, "Namede memory slot indices of the operating dataset are missing."

        # Parse loaded metadata.
        sort_columns, sort_rows = data["sorts"]

        # Load disambiguition sorting algorithms of data containers.
        self.sorts = (sort_columns, sort_rows)
        self.names = data["names"]
        return self
