# Import Python packages.
from typing import Any, List, Mapping, TypeVar

# Import external packages.
import pandas as pd

# Import developing packages.
from ....types import NPANYS
from ...base import BaseTransformPandas, ErrorTransformUnsupportPartial
from ..base import TransformCategoryEncode


# Type aliases.
Input = List[pd.DataFrame]
Output = List[pd.DataFrame]


# Self types.
SelfTransformDiscardEncodePandas = TypeVar(
    "SelfTransformDiscardEncodePandas", bound="TransformDiscardEncodePandas"
)


class TransformDiscardEncodePandas(BaseTransformPandas, TransformCategoryEncode[pd.DataFrame]):
    r"""
    Transformation for discarding categorical Pandas data.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "cateenc.discard.pandas"

    def input(self: SelfTransformDiscardEncodePandas, raw: Any, /) -> Input:
        r"""
        Convert raw data into input to the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Create an empty categorical container and an empty continuous container.
            return [
                pd.DataFrame([], columns=["feature-categorical"]),
                pd.DataFrame([], columns=["feature-continuous"]),
                pd.DataFrame([], columns=["label-categorical"]),
                pd.DataFrame([], columns=["label-continuous"]),
            ]
        else:
            # All the other cases are not supported.
            raise ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into input domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def output(self: SelfTransformDiscardEncodePandas, raw: Any, /) -> Output:
        r"""
        Convert raw data into output from the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Create an empty continuous container corresponding to null input.
            return [
                pd.DataFrame([], columns=[]),
                pd.DataFrame([], columns=["feature-continuous"]),
                pd.DataFrame([], columns=["label-categorical"]),
                pd.DataFrame([], columns=["label-continuous"]),
            ]
        else:
            # All the other cases are not supported.
            raise ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into output domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def transform(
        self: SelfTransformDiscardEncodePandas, input: Input, /, *args: Any, **kwargs: Any
    ) -> Output:
        r"""
        Transform input into output without inplacement.

        Args
        ----
        - input
            Input to the transformation.

        Returns
        -------
        - output
            Output from the transformation.
        """
        # Discard categorical data.
        _, continuous, *_labels = input
        return [pd.DataFrame([], columns=[], index=continuous.index), continuous, *_labels]

    def get_metadata(self: SelfTransformDiscardEncodePandas, /) -> Mapping[str, Any]:
        r"""
        Get metadata of the transformation.

        Args
        ----

        Returns
        -------
        - metadata
            Metadata of the transformation.
        """
        # Do nothing.
        return {}

    def get_numeric_data(self: SelfTransformDiscardEncodePandas, /) -> Mapping[str, NPANYS]:
        r"""
        Get numeric data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the transformation.
        """
        # Do nothing.
        return {}

    def get_alphabetic_data(self: SelfTransformDiscardEncodePandas, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # Do nothing.
        return {}

    def set_metadata(
        self: SelfTransformDiscardEncodePandas, metadata: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformDiscardEncodePandas:
        r"""
        Set metadata of the transformation.

        Args
        ----
        - metadata
            Metadata of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_numeric_data(
        self: SelfTransformDiscardEncodePandas, data: Mapping[str, NPANYS], /  # noqa: W504
    ) -> SelfTransformDiscardEncodePandas:
        r"""
        Set numeric data of the transformation.

        Args
        ----
        - data
            Numeric data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_alphabetic_data(
        self: SelfTransformDiscardEncodePandas, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformDiscardEncodePandas:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self
