# Import Python packages.
from typing import Any, List, Mapping, Optional, Sequence, TypeVar

# Import external packages.
import numpy as np
import pandas as pd
from sklearn.decomposition import PCA  # type: ignore[import-untyped]

# Import relatively from other modules.
from ....types import NPANYS
from ...base import BaseTransformPandas, ErrorTransformUnsupportPartial


# Type aliases.
Input = List[pd.DataFrame]
Output = List[pd.DataFrame]


# Self types.
SelfTransformPCAPandas = TypeVar("SelfTransformPCAPandas", bound="TransformPCAPandas")


class TransformPCAPandas(BaseTransformPandas):
    r"""
    Transformation for PCA on Pandas data.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "dimereduce.pca.pandas"

    def input(self: SelfTransformPCAPandas, raw: Any, /) -> Input:
        r"""
        Convert raw data into input to the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # To have the PCA running properly, we need at least two different samples with at least
            # one continuous feature.
            return [pd.DataFrame([[0.0], [1.0]], columns=["continuous"])]
        else:
            # All the other cases are not supported.
            raise ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into input domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def output(self: SelfTransformPCAPandas, raw: Any, /) -> Output:
        r"""
        Convert raw data into output from the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Expecting output is assumed to be default PCA with no reduction.
            return [
                pd.DataFrame(
                    PCA(n_components=1, random_state=42).fit_transform(np.array([[0.0], [1.0]])),
                    columns=["0"],
                )
            ]
        else:
            # All the other cases are not supported.
            raise ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into output domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def transform(
        self: SelfTransformPCAPandas,
        input: Input,
        /,
        *args: Any,
        columns: Optional[Sequence[str]] = None,
        **kwargs: Any,
    ) -> Output:
        r"""
        Transform input into output without inplacement.

        Args
        ----
        - input
            Input to the transformation.
        - columns
            Dataframe column titles after projection.
            If it is not given, it is the string representions of column integer indices.

        Returns
        -------
        - output
            Output from the transformation.
        """
        # Collect continuous dataframe to be projected.
        (dataframe,) = input

        # Utilize Scikit Learn handler directly.
        columns_ = list(columns) if columns else [str(i) for i in range(self.n_components)]
        return [pd.DataFrame(self.handler.transform(dataframe.values), columns=columns_)]

    def transform_(
        self: SelfTransformPCAPandas, input: Input, /, *args: Any, **kwargs: Any
    ) -> SelfTransformPCAPandas:
        r"""
        Transform input with inplacement.

        Args
        ----
        - input
            Input to the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Get the output and replace input of corresponding positions by the output.
        output = self.transform(input, *args, **kwargs)
        (dataframe,) = output
        input[0] = dataframe
        return self

    def fit(
        self: SelfTransformPCAPandas,
        input: Input,
        output: Output,
        /,
        *args: Any,
        scikit_learn_init_args: Sequence[Any] = [],
        scikit_learn_init_kwargs: Mapping[str, Any] = {},
        scikit_learn_fit_args: Sequence[Any] = [],
        scikit_learn_fit_kwargs: Mapping[str, Any] = {},
        **kwargs: Any,
    ) -> SelfTransformPCAPandas:
        r"""
        Fit transformation parameters by example input and output.

        Args
        ----
        - input
            Example input to the transformation.
        - output
            Example output from the transformation.
        - scikit_learn_init_args
            Positional arguments to PCA handler initialization of Scikit Learn package.
        - scikit_learn_init_kwargs
            Keyword arguments to PCA handler initialization of Scikit Learn package.
        - scikit_learn_fit_args
            Positional arguments to PCA handler parameter fitting of Scikit Learn package.
        - scikit_learn_fit_kwargs
            Keyword arguments to PCA handler parameter fitting of Scikit Learn package.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Get the dataframe to be handled.
        (dataframe,) = input

        # Make a clone of essential arguments, and perform auto correction and filling.
        scikit_learn_init_args_ = [*scikit_learn_init_args]
        scikit_learn_init_kwargs_ = {**scikit_learn_init_kwargs}
        scikit_learn_fit_args_ = [*scikit_learn_fit_args]
        scikit_learn_fit_kwargs_ = {**scikit_learn_fit_kwargs}
        assert (
            "n_components" in scikit_learn_init_kwargs_
        ), "Number of targeting dimensions must be explicitly defined."
        if (
            "random_state" not in scikit_learn_init_kwargs_
            or scikit_learn_init_kwargs_["random_state"] is None
        ):
            # Autofill random seed.
            scikit_learn_init_kwargs_["random_state"] = 42

        # Utilize Scikit Learn handler directly.
        self.scikit_learn_init_args = scikit_learn_init_args_
        self.scikit_learn_init_kwargs = scikit_learn_init_kwargs_
        self.handler = PCA(*self.scikit_learn_init_args, **self.scikit_learn_init_kwargs)
        self.handler.fit(dataframe.values, *scikit_learn_fit_args_, **scikit_learn_fit_kwargs_)
        self.n_components = self.handler.n_components_
        return self

    def get_metadata(self: SelfTransformPCAPandas, /) -> Mapping[str, Any]:
        r"""
        Get metadata of the transformation.

        Args
        ----

        Returns
        -------
        - metadata
            Metadata of the transformation.
        """
        # Do nothing.
        return {}

    def get_numeric_data(self: SelfTransformPCAPandas, /) -> Mapping[str, NPANYS]:
        r"""
        Get numeric data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the transformation.
        """
        # Collect all arraies that are not included in Scikit Learn parameters.
        return {
            "components": self.handler.components_,
            "explained_variance": self.handler.explained_variance_,
            "explained_variance_ratio": self.handler.explained_variance_ratio_,
            "singular_values": self.handler.singular_values_,
            "mean": self.handler.mean_,
        }

    def get_alphabetic_data(self: SelfTransformPCAPandas, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # Collect Scikit Learn parameters.
        return {
            "scikit_learn_init_args": self.scikit_learn_init_args,
            "scikit_learn_init_kwargs": self.scikit_learn_init_kwargs,
            "scikit_learn_params": self.handler.get_params(),
            "n_components": self.n_components,
        }

    def set_metadata(
        self: SelfTransformPCAPandas, metadata: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformPCAPandas:
        r"""
        Set metadata of the transformation.

        Args
        ----
        - metadata
            Metadata of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_numeric_data(
        self: SelfTransformPCAPandas, data: Mapping[str, NPANYS], /  # noqa: W504
    ) -> SelfTransformPCAPandas:
        r"""
        Set numeric data of the transformation.

        Args
        ----
        - data
            Numeric data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Safety check.
        assert "components" in data, "Numeric array of principal axes in feature space is missing."
        assert (
            "explained_variance" in data
        ), "Numeric array of variance explained by each of the selected components is missing."
        assert "explained_variance_ratio" in data, (
            "Numeric array of percentage of variance explained by each of the selected components"
            " is missing."
        )
        assert "singular_values" in data, (
            "Numeric array of singular values corresponding to each of the selected components is"
            " missing."
        )
        assert "mean" in data, "Numeric array of feature empirical mean is missing."

        # Cache loaded numeric data.
        self._content = data
        return self

    def set_alphabetic_data(
        self: SelfTransformPCAPandas, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformPCAPandas:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Safety check.
        assert (
            "scikit_learn_init_args" in data
        ), "Positional arguments of Scikit Learn handler initialization are missing."
        assert (
            "scikit_learn_init_kwargs" in data
        ), "Keyword arguments of Scikit Learn handler initialization are missing."
        assert "scikit_learn_params" in data, "Parameters of Scikit Learn handler are missing."
        assert "n_components" in data, "Number of targeting dimensions is missing."

        # Create a Scikit Learn handler of loaded numeric data.
        self.handler = PCA(*data["scikit_learn_init_args"], **data["scikit_learn_init_kwargs"])
        self.handler.set_params(**data["scikit_learn_params"])
        self.n_components = data["n_components"]

        # Update arraies of the handler by loaded numeric data.
        self.handler.components_ = self._content["components"]
        self.handler.explained_variance_ = self._content["explained_variance"]
        self.handler.explained_variance_ratio_ = self._content["explained_variance_ratio"]
        self.handler.singular_values_ = self._content["singular_values"]
        self.handler.mean_ = self._content["mean"]
        del self._content
        return self
