# Import Python packages.
import json
import os
from typing import Any, Mapping, Type, TypeVar

# Import relatively from other modules.
from .base import _REGISTRATIONS, BaseTransform


# Self types.
SelfFactoryTransform = TypeVar("SelfFactoryTransform", bound="FactoryTransform")


class FactoryTransform(object):
    r"""
    Factory producing transformations.
    """

    def get_class(self: SelfFactoryTransform, identifier: str, /) -> Type[BaseTransform[Any, Any]]:
        r"""
        Get a transformation class from identifier.

        Args
        ----
        - identifier
            Identifier of the transformation.

        Returns
        -------
        - cls
            A transformation class.
        """
        # Get the transformation class from registry.
        assert (
            identifier in _REGISTRATIONS
        ), f'Unregistered transformation identifier: "{identifier:s}".'
        return _REGISTRATIONS[identifier]

    def from_args(
        self: SelfFactoryTransform, identifier: str, /, *args: Any, **kwargs: Any
    ) -> BaseTransform[Any, Any]:
        r"""
        Create a transformation directly from argumnets.

        Args
        ----
        - identifier
            Identifier of the transformation.

        Returns
        -------
        - data
            A transformation.
        """
        # Create an instance based on arguments.
        return self.get_class(identifier)(*args, **kwargs)

    def from_metadata(
        self: SelfFactoryTransform, metadata: Mapping[str, Any], /  # noqa: W504
    ) -> BaseTransform[Any, Any]:
        r"""
        Create a transformation directly from metadata.

        Args
        ----
        - metadata
            Metadata of transformation.

        Returns
        -------
        - transform
            A transformation.
        """
        # Collect essential information for creating the transformation.
        assert "_identifier" in metadata, 'Missing metadata: "_identifier".'
        assert "_children" in metadata, 'Missing metadata: "_children".'
        identifier = metadata["_identifier"]
        children = metadata["_children"]

        # Create an instance based on metadata and metadata of children.
        return self.get_class(identifier)(
            children={
                child_name: self.from_metadata(child_metadata)
                for child_name, child_metadata in children.items()
            },
            **metadata,
        )

    def load_metadata(self: SelfFactoryTransform, path: str, /) -> Mapping[str, Any]:
        r"""
        Load only metadata recursively from given path.

        Args
        ----
        - path
            Path of metadata.

        Returns
        -------
        - metadata
            Loaded metadata of a transformation.
        """
        # Collect metadata of current level.
        metadata: Mapping[str, Any]
        with open(os.path.join(path, "metadata.json"), "r") as file:
            # Load metadata in JSON format.
            metadata = json.load(file)
        assert "_children" not in metadata

        # Collect metadata of children levels recursively.
        with open(os.path.join(path, "children.json"), "r") as file:
            # Load children names as loading index.
            names = json.load(file)
        metadata = {
            "_children": {name: self.load_metadata(os.path.join(path, name)) for name in names},
            **metadata,
        }
        return metadata

    def from_filesys(self: SelfFactoryTransform, path: str, /) -> BaseTransform[Any, Any]:
        r"""
        Create a transformation directly from data on file system.

        Args
        ----
        - path
            Root path of transformation data.

        Returns
        -------
        - transform
            A transformation.
        """
        # Load metadata, create instance from metadata, and overwrite instance data by data from
        # file system.
        metadata = self.load_metadata(path)
        transform = self.from_metadata(metadata)
        transform.load(path)
        return transform
