# Import Python packages.
from typing import Any, List, Mapping, Optional, Sequence, TypeVar

# Import external packages.
import numpy as np
import pandas as pd
from sklearn.preprocessing import QuantileTransformer  # type: ignore[import-untyped]

# Import relatively from other modules.
from ....types import NPANYS
from ...base import BaseTransformPandas, ErrorTransformUnsupportPartial


# Type aliases.
Input = List[pd.DataFrame]
Output = List[pd.DataFrame]


# Self types.
SelfTransformQuantilizePandas = TypeVar(
    "SelfTransformQuantilizePandas", bound="TransformQuantilizePandas"
)


class TransformQuantilizePandas(BaseTransformPandas):
    r"""
    Transformation for quantile normalization on Pandas data.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "normalize.quantile.pandas"

    def input(self: SelfTransformQuantilizePandas, raw: Any, /) -> Input:
        r"""
        Convert raw data into input to the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # To have the PCA running properly, we need at least two different samples with at least
            # one continuous feature.
            return [pd.DataFrame([[0.0], [1.0]], columns=["continuous"])]
        else:
            # All the other cases are not supported.
            raise ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into input domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def output(self: SelfTransformQuantilizePandas, raw: Any, /) -> Output:
        r"""
        Convert raw data into output from the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Expecting output is assumed to be default PCA with no reduction.
            return [
                pd.DataFrame(
                    QuantileTransformer(n_quantiles=2, random_state=42).fit_transform(
                        np.array([[0.0], [1.0]])
                    ),
                    columns=["continuous"],
                )
            ]
        else:
            # All the other cases are not supported.
            raise ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into output domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def transform(
        self: SelfTransformQuantilizePandas,
        input: Input,
        /,
        *args: Any,
        columns: Optional[Sequence[str]] = None,
        **kwargs: Any,
    ) -> Output:
        r"""
        Transform input into output without inplacement.

        Args
        ----
        - input
            Input to the transformation.
        - columns
            Dataframe column titles after projection.
            If it is not given, it is the string representions of column integer indices.

        Returns
        -------
        - output
            Output from the transformation.
        """
        # Collect continuous dataframe to be projected.
        (dataframe,) = input

        # Utilize Scikit Learn handler directly.
        return [pd.DataFrame(self.handler.transform(dataframe.values), columns=dataframe.columns)]

    def transform_(
        self: SelfTransformQuantilizePandas, input: Input, /, *args: Any, **kwargs: Any
    ) -> SelfTransformQuantilizePandas:
        r"""
        Transform input with inplacement.

        Args
        ----
        - input
            Input to the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Get the output and replace input of corresponding positions by the output.
        output = self.transform(input, *args, **kwargs)
        (dataframe,) = output
        input[0] = dataframe
        return self

    def inverse(
        self: SelfTransformQuantilizePandas, output: Output, /, *args: Any, **kwargs: Any
    ) -> Input:
        r"""
        Inverse output into input without inplacement.

        Args
        ----
        - output
            Output from the transformation.

        Returns
        -------
        - input
            Input to the transformation.
        """
        # Collect continuous dataframe to be recovered.
        (dataframe,) = output

        # Utilize Scikit Learn handler directly.
        return [
            pd.DataFrame(
                self.handler.inverse_transform(dataframe.values), columns=dataframe.columns
            )
        ]

    def inverse_(
        self: SelfTransformQuantilizePandas, output: Output, /, *args: Any, **kwargs: Any
    ) -> SelfTransformQuantilizePandas:
        r"""
        Inverse output with inplacement.

        Args
        ----
        - output
            Output from the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Get the input and replace output of corresponding positions by the input.
        input = self.inverse(output, *args, **kwargs)
        (dataframe,) = input
        output[0] = dataframe
        return self

    def fit(
        self: SelfTransformQuantilizePandas,
        input: Input,
        output: Output,
        /,
        *args: Any,
        scikit_learn_init_args: Sequence[Any] = [],
        scikit_learn_init_kwargs: Mapping[str, Any] = {},
        scikit_learn_fit_args: Sequence[Any] = [],
        scikit_learn_fit_kwargs: Mapping[str, Any] = {},
        **kwargs: Any,
    ) -> SelfTransformQuantilizePandas:
        r"""
        Fit transformation parameters by example input and output.

        Args
        ----
        - input
            Example input to the transformation.
        - output
            Example output from the transformation.
        - scikit_learn_init_args
            Positional arguments to PCA handler initialization of Scikit Learn package.
        - scikit_learn_init_kwargs
            Keyword arguments to PCA handler initialization of Scikit Learn package.
        - scikit_learn_fit_args
            Positional arguments to PCA handler parameter fitting of Scikit Learn package.
        - scikit_learn_fit_kwargs
            Keyword arguments to PCA handler parameter fitting of Scikit Learn package.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Get the dataframe to be handled.
        (dataframe,) = input

        # Make a clone of essential arguments, and perform auto correction and filling.
        scikit_learn_init_args_ = [*scikit_learn_init_args]
        scikit_learn_init_kwargs_ = {**scikit_learn_init_kwargs}
        scikit_learn_fit_args_ = [*scikit_learn_fit_args]
        scikit_learn_fit_kwargs_ = {**scikit_learn_fit_kwargs}
        assert (
            "n_quantiles" in scikit_learn_init_kwargs_
        ), "Number of targeting quantiles must be explicitly defined."
        scikit_learn_init_kwargs_["n_quantiles"] = min(
            scikit_learn_init_kwargs_["n_quantiles"], len(dataframe)
        )
        if (
            "random_state" not in scikit_learn_init_kwargs_
            or scikit_learn_init_kwargs_["random_state"] is None
        ):
            # Autofill random seed.
            scikit_learn_init_kwargs_["random_state"] = 42
        if (
            "subsample" not in scikit_learn_init_kwargs_
            or scikit_learn_init_kwargs_["subsample"] is None
        ):
            # Autofill random seed.
            scikit_learn_init_kwargs_["subsample"] = len(dataframe)

        # Utilize Scikit Learn handler directly.
        self.scikit_learn_init_args = scikit_learn_init_args_
        self.scikit_learn_init_kwargs = scikit_learn_init_kwargs_
        self.handler = QuantileTransformer(
            *self.scikit_learn_init_args, **self.scikit_learn_init_kwargs
        )
        self.handler.fit(dataframe.values, *scikit_learn_fit_args_, **scikit_learn_fit_kwargs_)
        return self

    def get_metadata(self: SelfTransformQuantilizePandas, /) -> Mapping[str, Any]:
        r"""
        Get metadata of the transformation.

        Args
        ----

        Returns
        -------
        - metadata
            Metadata of the transformation.
        """
        # Do nothing.
        return {}

    def get_numeric_data(self: SelfTransformQuantilizePandas, /) -> Mapping[str, NPANYS]:
        r"""
        Get numeric data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the transformation.
        """
        # Collect all arraies that are not included in Scikit Learn parameters.
        return {"quantiles": self.handler.quantiles_, "references": self.handler.references_}

    def get_alphabetic_data(self: SelfTransformQuantilizePandas, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # Collect Scikit Learn parameters.
        return {
            "scikit_learn_init_args": self.scikit_learn_init_args,
            "scikit_learn_init_kwargs": self.scikit_learn_init_kwargs,
            "scikit_learn_params": self.handler.get_params(),
        }

    def set_metadata(
        self: SelfTransformQuantilizePandas, metadata: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformQuantilizePandas:
        r"""
        Set metadata of the transformation.

        Args
        ----
        - metadata
            Metadata of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_numeric_data(
        self: SelfTransformQuantilizePandas, data: Mapping[str, NPANYS], /  # noqa: W504
    ) -> SelfTransformQuantilizePandas:
        r"""
        Set numeric data of the transformation.

        Args
        ----
        - data
            Numeric data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Safety check.
        assert "quantiles" in data, (
            "Numeric array of raw feature values corresponding the quantiles of reference is"
            " missing."
        )
        assert (
            "references" in data
        ), "Numeric array of normalized quantiles of reference is missing."

        # Cache loaded numeric data.
        self._content = data
        return self

    def set_alphabetic_data(
        self: SelfTransformQuantilizePandas, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformQuantilizePandas:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Safety check.
        assert (
            "scikit_learn_init_args" in data
        ), "Positional arguments of Scikit Learn handler initialization are missing."
        assert (
            "scikit_learn_init_kwargs" in data
        ), "Keyword arguments of Scikit Learn handler initialization are missing."
        assert "scikit_learn_params" in data, "Parameters of Scikit Learn handler are missing."

        # Create a Scikit Learn handler of loaded numeric data.
        self.handler = QuantileTransformer(
            *data["scikit_learn_init_args"], **data["scikit_learn_init_kwargs"]
        )
        self.handler.set_params(**data["scikit_learn_params"])

        # Update arraies of the handler by loaded numeric data.
        self.handler.quantiles_ = self._content["quantiles"]
        self.handler.references_ = self._content["references"]
        del self._content
        return self
