# Import Python packages.
import copy
import os
from typing import Any, Callable, Mapping, Optional, Sequence, Tuple, TypeVar, get_type_hints

# Import external packages.
from typeguard import check_type

# Import developing library.
import fin_tech_py_toolkit as lib


# Type variables.
Content = TypeVar("Content")


def template_test_data(
    root: str,
    identifier: str,
    factory: lib.data.FactoryData,
    content: Content,
    eq: Callable[[Content, Content], bool],
    modify: Callable[[Content], None],
    /,
    *,
    allow_alias: bool = False,
    check_identity: Optional[
        Callable[[lib.data.BaseData[Content], lib.data.BaseData[Content]], bool]
    ] = None,
    init_args: Sequence[Any] = [],
    init_kwargs: Mapping[str, Any] = {},
    numeric_translate: Tuple[
        Tuple[Sequence[str], Mapping[str, Any]], Tuple[Sequence[str], Mapping[str, Any]]
    ] = (([], {}), ([], {})),
    potential_translates: Sequence[
        Tuple[str, Tuple[Sequence[str], Mapping[str, Any]], Tuple[Sequence[str], Mapping[str, Any]]]
    ] = [],
    allow_translate_default: bool = True,
) -> None:
    r"""
    Template of testing data container.

    Args
    ----
    - root
        File system storage root.
    - identifier
        Testing data container identifier.
    - factory
        Factory used to reproduce testing data container instance.
    - content
        Source content of the data container.
    - eq
        Equality comparator on content domain.
    - modify
        Original content modifier.
    - allow_alias
        If True, the content of data container is an alias of source content.
        If False, the content of data container is a completed copy of source content.
    - check_identity
        Chech the validation of identity generation of the data container.
        If not given, generated identity element is assumed to be valid.
    - init_args
        Positional arguments for testing initialization.
    - init_kwargs
        Keyword arguments for testing initialization.
    - numeric_translate
        Positional and keyword arguments for testing translation between numeric data and the data
        container.
    - potential_translates
        Potentially compatible storage formats for the content in the data container, and their
        positional and keyword arguments for translation between them and the data container.
    - allow_translate_default
        If True, allow using initialization arguments as default for translation from other formats
        to the data container.
        If False, all translation arguments must be explicitly provided.

    Returns
    -------
    """
    # Finalize runtime flags.
    data = factory.from_args(
        identifier, content, *init_args, allow_alias=allow_alias, **init_kwargs
    )
    tags = data.tags
    test_identity = tags.emptiable
    test__disambiguition = not tags.ambiguous

    # Run test snippets based on flags.
    if test_identity:
        # Generate identity element.
        identity = data.identity()
        assert check_identity is None or check_identity(data, identity)

    # Run test snippets based on flags.
    if test__disambiguition:
        # Explicitly generate an unambiguous data container, and compare with the data container.
        data_ = factory.from_args(
            identifier, content, *init_args, allow_alias_disambiguition=allow_alias, **init_kwargs
        )
        data_ = data_._disambiguate(deep=not data_.allow_alias_disambiguition)
        assert eq(data._content, data_._content)
        assert data.hashcode == data_.hashcode

    # Modify source content, and compare with saved content of container.
    path = os.path.join(root, "data.json")
    content1 = content
    content2 = copy.deepcopy(data._content)
    modify(content1)
    data.save(path)
    data.load(path)
    assert eq(data._content, content1 if allow_alias else content2)

    # Traverse every translation, where numeric translation will always be checked first.
    for form, (to_args, to_kwargs), (from_args_, from_kwargs_) in [
        ("numeric", *numeric_translate),
        *potential_translates,
    ]:
        # Check translation annotation, and stability.
        to_translate = getattr(data, "to_" + form)
        from_translate = getattr(data, "from_" + form)
        annotation = get_type_hints(to_translate)["return"]
        if allow_translate_default:
            # Auto fill arguments by initialization arguments.
            from_args = [
                (from_args_[i] if i < len(from_args_) else init_args[i])
                for i in range(max(len(init_args), len(from_args_)))
            ]
            from_kwargs = {**init_kwargs, **from_kwargs_}
        translation = to_translate(*to_args, **to_kwargs)
        data_ = from_translate(translation, *from_args, **from_kwargs)
        check_type(translation, annotation)
        assert eq(data_._content, data._content)
        assert data_.hashcode == data.hashcode
