# Import Python packages.
import json
from typing import Any, List, Mapping, Optional, Sequence, Tuple, Type, TypeVar

# Import developing library.
import fin_tech_py_toolkit as lib
from fin_tech_py_toolkit.types import NPANYS

# Import testing library.
from ..data.supp_base import DataInts


# Type aliases.
Input = Sequence[Tuple[str, Optional[str]]]
Output = List[DataInts]


# Self types.
SelfDatasetInts = TypeVar("SelfDatasetInts", bound="DatasetInts")


class DatasetInts(lib.datasets.BaseDataset[DataInts]):
    r"""
    Dataset of integers.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "dataset.tabular.test_ints"

    @classmethod
    def from_memalias(
        cls: Type[SelfDatasetInts],
        memory: Output,
        memory_names: Sequence[str],
        /,
        *args: Any,
        addresses: Mapping[str, str] = {},
        **kwargs: Any,
    ) -> SelfDatasetInts:
        r"""
        Initialize the class from direct data memory aliasing.

        Args
        ----
        - memory
            Data memory.
        - memory_names
            Named indices of memory slots.
        - addresses
            Named addresses corresponding to memory slots respectively.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Create an empty dataset first.
        dataset = cls(*args, **kwargs)

        # Alias data memory.
        return dataset.memalias(memory, memory_names, *args, addresses=addresses, **kwargs)

    def memorize(
        self: SelfDatasetInts,
        addresses: Sequence[Tuple[str, Optional[str]]],
        /,
        *args: Any,
        cache_read: bool = True,
        cache_relpaths: Mapping[str, str] = {},
        **kwargs: Any,
    ) -> SelfDatasetInts:
        r"""
        Load structed content from addresses into memory.

        Args
        ----
        - addresses
            Named addresses of essential data.
        - cache_read
            If True, non-local data (e.g., from URLs) will be cached into class cache directory on
            local file system.
        - cache_relpaths
            Relative local paths of caching data w.r.t. class cache directory.
            Index names should match with names from true addresses.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Read integers from file system.
        addresses_ = {}
        memory = []
        memory_names = []
        memory_indices = {}
        for i, (name, address) in enumerate(addresses):
            # Must read integers from file system during memorization process.
            assert address is not None, f'Address for integer data "{name:s}" is missing.'
            addresses_[name] = address

            # Read integers from JSON file, and create a data container.
            with open(address, "r") as file:
                # Read integers.
                ints = json.load(file)
            data = DataInts(ints)
            memory.append(data)
            memory_names.append(name)
            memory_indices[name] = i

        # Store memory related information in immutable format.
        self._addresses = addresses_
        self.memory = memory
        self.memory_names = memory_names
        self._memory_indices = memory_indices
        return self

    def get_numeric_data(self: SelfDatasetInts, /) -> Mapping[str, NPANYS]:
        r"""
        Get numeric data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the transformation.
        """
        # Do nothing.
        return {}

    def get_alphabetic_data(self: SelfDatasetInts, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # Do nothing.
        return {}

    def set_numeric_data(
        self: SelfDatasetInts, data: Mapping[str, NPANYS], /  # noqa: W504
    ) -> SelfDatasetInts:
        r"""
        Set numeric data of the transformation.

        Args
        ----
        - data
            Numeric data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_alphabetic_data(
        self: SelfDatasetInts, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfDatasetInts:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self
