# Import Python packages.
import os
from typing import Any, Callable, Mapping, Optional, Sequence, Tuple, TypeVar, cast

# Import developing library.
import fin_tech_py_toolkit as lib


# Type variables.
Content = TypeVar("Content")


def template_test_dataset(
    root: str,
    identifier: str,
    factory: lib.transforms.FactoryTransform,
    sources: Sequence[Content],
    addresses: Sequence[Tuple[str, Optional[str]]],
    eq: Callable[[Content, Content], bool],
    /,
    *,
    link: bool = True,
    memorize_args: Sequence[Any] = [],
    memorize_kwargs: Mapping[str, Any] = {},
) -> None:
    r"""
    Template of testing dataset.

    Args
    ----
    - root
        File system storage root.
    - identifier
        Testing dataset identifier.
    - factory
        Factory used to reproduce testing transformation (dataset) instance.
    - sources
        Source data corresponding to dataset memory respectively.
    - addresses
        Testing addresses from where dataset memory is loaded.
    - eq
        Equality comparator on data domain.
    - link
        If True, the dataset will be treated as a link of original data, thus when we save the
        dataset, it will only save addresses of original data.
        If False, the dataset will create a copy of original data under its own format for
        saving.
    - memorize_args
        Positional arguments for testing loading data into memory.
    - memorize_kwargs
        Keyword arguments for testing loading data into memory.

    Returns
    -------
    """
    # Reserve essential temporary directories.
    root_cache = os.path.join(root, "cache")
    params = os.path.join(root, "parameters")
    assert not os.path.isdir(root_cache)
    assert not os.path.isdir(params)

    # Define free-to-access address.
    addresses_ = [(name, None) for name, _ in addresses]

    # Basic way of creation.
    # It is the alias of create-then-fit.
    dataset = cast(
        lib.datasets.BaseDataset[lib.data.BaseData[Content]], factory.get_class(identifier)
    ).from_storage(addresses, *memorize_args, cache_prefix=root_cache, link=link, **memorize_kwargs)
    memory = dataset.transform(addresses_ if memorize_kwargs["cache_read"] else addresses)
    for i in range(max(len(memory), len(sources))):
        # All content in this test is the same table.
        assert eq(memory[i]._content, sources[i])
    dataset.save(params)

    # Another basic way of dataset creation.
    dataset_ = dataset.__class__.from_memalias(
        *dataset.get_memalias_args(), **dataset.get_memalias_kwargs()
    )
    memory_ = dataset_.transform(addresses_)
    for i in range(max(len(memory_), len(memory))):
        # All content in this test is the same table.
        assert eq(memory_[i]._content, memory[i]._content)

    # Generate a dataset through factory.
    dataset_ = cast(
        lib.datasets.BaseDataset[lib.data.BaseData[Content]], factory.from_filesys(params)
    )
    memory_ = dataset_.transform(addresses_)
    for i in range(max(len(memory_), len(memory))):
        # All content in this test is the same table.
        assert eq(memory_[i]._content, memory[i]._content)
