# Import Python packages.
import os

# Import PyTest packages.
import pytest

# Import PyTest external packages.
from py._path.local import LocalPath

# Import developing library.
import fin_tech_py_toolkit as lib


def test_rmtree(*, tmpdir: LocalPath) -> None:
    r"""
    Test remove a directory.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Create a temporary directory to be removed.
    pypath = tmpdir.mkdir("directory").join("file")  # type: ignore[no-untyped-call]
    pypath.write("content")
    root = str(tmpdir)

    # Remove the directory.
    assert os.path.isdir(os.path.join(root, "directory"))
    assert os.path.isfile(os.path.join(root, "directory", "file"))
    lib.io.filesys.rmtree(os.path.join(root, "directory"))
    assert not os.path.isdir(os.path.join(root, "directory"))
    assert not os.path.isfile(os.path.join(root, "directory", "file"))


def test_rmfile(*, tmpdir: LocalPath) -> None:
    r"""
    Test remove a file.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Create a temporary file to be removed.
    pypath = tmpdir.join("file")  # type: ignore[no-untyped-call]
    pypath.write("content")
    root = str(tmpdir)

    # Remove the directory.
    assert os.path.isfile(os.path.join(root, "file"))
    lib.io.filesys.rmfile(os.path.join(root, "file"))
    assert not os.path.isfile(os.path.join(root, "file"))


@pytest.mark.parametrize(
    ("must_be_new", "allow_auto_remove"),
    [
        pytest.param(True, True, id="auto-remove"),
        pytest.param(
            True, False, marks=[pytest.mark.xfail(raises=FileExistsError)], id="dup-error"
        ),
        pytest.param(False, True, id="silent-reuse-improper"),
        pytest.param(False, False, id="silent-reuse"),
    ],
)
def test_mkdirs(*, tmpdir: LocalPath, must_be_new: bool, allow_auto_remove: bool) -> None:
    r"""
    Test making a directory.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.
    - must_be_new
        If True, the directory must be newly made, thus removal is required if it already exists.
        Otherwise, it will do nothing if the directory already exists.
    - allow_auto_remove
        If True, automatic removal is allowed before directory making.
        Otherwise, removal must be explicitly called before calling this function.

    Returns
    -------
    """
    # Create a temporary directory for duplicate.
    pypath = tmpdir.mkdir("directory").join("file")  # type: ignore[no-untyped-call]
    pypath.write("content")
    root = str(tmpdir)

    # Make directory with duplicate being properly handled according to arguments.
    assert os.path.isdir(os.path.join(root, "directory"))
    assert os.path.isfile(os.path.join(root, "directory", "file"))
    lib.io.filesys.mkdirs(
        os.path.join(root, "directory"),
        must_be_new=must_be_new,
        allow_auto_remove=allow_auto_remove,
        timeout=1e-3,
        interval=1e-4,
    )
    is_previous = os.path.isfile(os.path.join(root, "directory", "file"))
    assert os.path.isdir(os.path.join(root, "directory"))
    assert (must_be_new and not is_previous) or (not must_be_new and is_previous)
