# Import Python packages.
import math
from typing import Any, List, Mapping, Sequence, Tuple, TypeVar

# Import external packages.
import numpy as np

# Import developing library.
import fin_tech_py_toolkit as lib

# Import testing library.
from ...data.supp_base import DataInts
from ...datasets.supp_base import DatasetInts


# Type aliases.
Input = List[DatasetInts]
Output = List[DatasetInts]


# Self types.
SelfTransdatasetSplitInts = TypeVar("SelfTransdatasetSplitInts", bound="TransdatasetSplitInts")


class TransdatasetSplitInts(lib.transdatasets.TransdatasetSplit[DatasetInts]):
    r"""
    Transformation for splitting an integer dataset into multiple integer datasets.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "split.dataset.test_ints"

    def input(self: SelfTransdatasetSplitInts, raw: Any, /) -> Input:
        r"""
        Convert raw data into input to the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Nothing to be unraveled.
            return [
                DatasetInts.from_memalias(
                    [DataInts.from_numeric({"content": np.array([], dtype=np.int64)})], ["full"]
                )
            ]
        else:
            # All the other cases are not supported.
            raise lib.transforms.ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into input domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def output(self: SelfTransdatasetSplitInts, raw: Any, /) -> Output:
        r"""
        Convert raw data into output from the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Nothing to be unraveled.
            return [
                DatasetInts.from_memalias(
                    [DataInts.from_numeric({"content": np.array([], dtype=np.int64)})], ["full"]
                ),
                DatasetInts.from_memalias(
                    [DataInts.from_numeric({"content": np.array([], dtype=np.int64)})], ["full"]
                ),
            ]
        else:
            # All the other cases are not supported.
            raise lib.transforms.ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into output domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def transform(
        self: SelfTransdatasetSplitInts,
        input: Input,
        /,
        *args: Any,
        props: Sequence[Tuple[Sequence[str], Tuple[int, int]]] = [],
        **kwargs: Any,
    ) -> Output:
        r"""
        Transform input into output without inplacement.

        Args
        ----
        - input
            Input to the transformation.
        - props
            Proportions of major and minor split halves at each memory slot.

        Returns
        -------
        - output
            Output from the transformation.
        """
        # Parse input memory.
        (dataset,) = input

        # Collect proportions of both halves.
        props_ = self._collect_props(props)

        # Collect content of heading and tailing halves.
        contents_head = []
        contents_tail = []
        names = []
        for data, name in zip(dataset.memory, dataset.memory_names):
            # Collect breakpoint between heading and tailing halves.
            # Content is more preferred towards headning.
            content = data._content
            (prop_head, prop_tail) = props_[name]
            size = len(content)
            threshold = int(
                math.ceil(float(size) * float(prop_head) / float(prop_head + prop_tail))
            )
            contents_head.append(content[:threshold])
            contents_tail.append(content[threshold:])
            names.append(name)

        # Create output datasets.
        datasets = [
            DatasetInts.from_memalias(
                [
                    DataInts(content, allow_alias_disambiguition=self.allow_alias)
                    for content in contents
                ],
                names,
            )
            for contents in [contents_head, contents_tail]
        ]
        return datasets

    def get_alphabetic_data(self: SelfTransdatasetSplitInts, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # Do nothing.
        return {}

    def set_alphabetic_data(
        self: SelfTransdatasetSplitInts, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransdatasetSplitInts:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self
