# Import Python packages.
from typing import Any, Callable, List, Mapping, Tuple, cast

# Import external packages.
import pandas as pd

# Import PyTest packagtes.
import pytest

# Import PyTest external packages.
from py._path.local import LocalPath

# Import developing library.
import fin_tech_py_toolkit as lib

from ...transforms.utils import template_test_io, template_test_transform

# Import testing library.
from ...utils import eq_dataframe, to_eq_data, to_eq_dataset, to_eq_plural_ordered


# Type aliases.
Input = List[lib.datasets.DatasetTabular]
Output = List[pd.DataFrame]


# Runtime constants.
IDENTIFIER = lib.transdatasets.TransdatasetUnravelTabular._IDENTIFIER


# Cast comparator.
eq_dataset_tabular = cast(
    Callable[[List[lib.datasets.DatasetTabular], List[lib.datasets.DatasetTabular]], bool],
    to_eq_plural_ordered(to_eq_dataset(to_eq_data(eq_dataframe))),
)


def synthesize() -> Tuple[Tuple[Input, Output], Input, Output, Mapping[str, Any]]:
    r"""
    Synthesize test I/O.

    Args
    ----

    Returns
    -------
    - example
        Input and output examples.
    - input
        Input case.
    - output
        Output case.
    - supplement
        Supplementary materical for synthesized test.
    """
    # Create a simple example.
    dataframe = pd.DataFrame({"int": [3, 2, 1], "float": [3.0, 2.0, 1.0]})
    input: Input
    input = [
        lib.datasets.DatasetTabularSimple.from_memalias(
            [lib.data.DataTabular(dataframe, sort_columns="alphabetic", sort_rows="rankable")],
            ["full"],
            sorts=("alphabetic", "rankable"),
        )
    ]
    output: Output
    output = [dataframe.copy()]
    return (input, output), input, output, {}


@pytest.mark.parametrize(
    ("raw_input", "raw_output"),
    [
        pytest.param(
            ...,
            None,
            id="unsupport-input",
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        ),
        pytest.param(
            None,
            ...,
            id="unsupport-output",
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        ),
        pytest.param(None, None, id="both-null"),
    ],
)
def test_io(*, raw_input: Any, raw_output: Any) -> None:
    r"""
    Test transformation input and output domain formalization.

    Args
    ----
    - raw_input
        Raw input.
    - raw_output
        Raw output.

    Returns
    -------
    """
    # Initialize testing transformation.
    factory = lib.transforms.FactoryTransform()

    # Run test template.
    template_test_io(
        IDENTIFIER,
        factory,
        raw_input,
        raw_output,
        eq_dataset_tabular,
        to_eq_plural_ordered(eq_dataframe),
    )


@pytest.mark.parametrize(
    "inverse_kwargs",
    [
        pytest.param({}, id="common"),
        pytest.param(dict(sort_columns="identity", sort_rows="identity"), id="ambiguous-inverse"),
    ],
)
def test_default(*, tmpdir: LocalPath, inverse_kwargs: Mapping[str, Any]) -> None:
    r"""
    Test unravel dataset transformation for tabular dataset.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.
    - inverse_kwargs
        Keyword arguments for testing inversion.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.transforms.FactoryTransform()

    # Generate inputs and outputs.
    example, input, output, supplement = synthesize()

    # Run test template.
    template_test_transform(
        root,
        IDENTIFIER,
        factory,
        example,
        input,
        output,
        eq_dataset_tabular,
        to_eq_plural_ordered(eq_dataframe),
        inverse_kwargs=inverse_kwargs,
    )


def test_empty(*, tmpdir: LocalPath) -> None:
    r"""
    Test unravel dataset transformation for empty tabular dataset.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.transforms.FactoryTransform()

    # Generate inputs and outputs.
    transform = factory.from_args(IDENTIFIER)
    input = transform.input(None)
    output = transform.output(None)
    example: Tuple[Input, Output]
    example = (input, [])

    # Run test template.
    template_test_transform(
        root,
        IDENTIFIER,
        factory,
        example,
        input,
        output,
        eq_dataset_tabular,
        to_eq_plural_ordered(eq_dataframe),
    )
