# Import Python packages.
from typing import Any, Mapping, TypeVar, cast

# Import developing library.
import fin_tech_py_toolkit as lib
from fin_tech_py_toolkit.types import NPANYS


# Self types.
SelfTransformNull = TypeVar("SelfTransformNull", bound="TransformNull")
SelfTransformIdentityComposition = TypeVar(
    "SelfTransformIdentityComposition", bound="TransformIdentityComposition"
)


class TransformNull(lib.transforms.BaseTransform[Any, Any]):
    r"""
    Transformation with no functionality.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "test_null"

    def input(self: SelfTransformNull, raw: Any, /) -> Any:
        r"""
        Convert raw data into input to the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Do nothing.
        return raw

    def output(self: SelfTransformNull, raw: Any, /) -> Any:
        r"""
        Convert raw data into output from the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Do nothing.
        return raw

    def transform(self: SelfTransformNull, input: Any, /, *args: Any, **kwargs: Any) -> Any:
        r"""
        Transform input into output without inplacement.

        Args
        ----
        - input
            Input to the transformation.

        Returns
        -------
        - output
            Output from the transformation.
        """
        # Do nothing.
        return input

    def get_metadata(self: SelfTransformNull, /) -> Mapping[str, Any]:
        r"""
        Get metadata of the transformation.

        Args
        ----

        Returns
        -------
        - metadata
            Metadata of the transformation.
        """
        # No metadata is defined.
        return {}

    def get_numeric_data(self: SelfTransformNull, /) -> Mapping[str, NPANYS]:
        r"""
        Get numeric data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the transformation.
        """
        # No numeric data is defined.
        return {}

    def get_alphabetic_data(self: SelfTransformNull, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # No alphabetic data is defined.
        return {}

    def set_metadata(self: SelfTransformNull, metadata: Mapping[str, Any], /) -> SelfTransformNull:
        r"""
        Set metadata of the transformation.

        Args
        ----
        - metadata
            Metadata of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_numeric_data(
        self: SelfTransformNull, data: Mapping[str, NPANYS], /  # noqa: W504
    ) -> SelfTransformNull:
        r"""
        Set numeric data of the transformation.

        Args
        ----
        - data
            Numeric data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_alphabetic_data(
        self: SelfTransformNull, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformNull:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self


class TransformIdentityComposition(lib.transforms.BaseTransform[Any, Any]):
    r"""
    Transformation for identity composition.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "test_identity_composition"

    def input(self: SelfTransformIdentityComposition, raw: Any, /) -> Any:
        r"""
        Convert raw data into input to the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Do nothing.
        return raw

    def output(self: SelfTransformIdentityComposition, raw: Any, /) -> Any:
        r"""
        Convert raw data into output from the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Do nothing.
        return raw

    def transform(
        self: SelfTransformIdentityComposition, input: Any, /, *args: Any, **kwargs: Any
    ) -> Any:
        r"""
        Transform input into output without inplacement.

        Args
        ----
        - input
            Input to the transformation.

        Returns
        -------
        - output
            Output from the transformation.
        """
        # Simply apply identity children in order.
        identity1 = self.get_child_identity1()
        identity2 = self.get_child_identity2()
        hidden = input
        hidden = identity1.transform(hidden)
        hidden = identity2.transform(hidden)
        return hidden

    def transform_(
        self: SelfTransformIdentityComposition, input: Any, /, *args: Any, **kwargs: Any
    ) -> SelfTransformIdentityComposition:
        r"""
        Transform input with inplacement.

        Args
        ----
        - input
            Input to the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Simply apply identity children in order.
        identity1 = self.get_child_identity1()
        identity2 = self.get_child_identity2()
        identity1.transform_(input)
        identity2.transform_(input)
        return self

    def inverse(
        self: SelfTransformIdentityComposition, output: Any, /, *args: Any, **kwargs: Any
    ) -> Any:
        r"""
        Inverse output back into input without inplacement.

        Args
        ----
        - output
            Output from the transformation.

        Returns
        -------
        - input
            Input to the transformation.
        """
        # Simply apply identity children in order.
        identity1 = self.get_child_identity1()
        identity2 = self.get_child_identity2()
        hidden = output
        hidden = identity2.transform(hidden)
        hidden = identity1.transform(hidden)
        return hidden

    def inverse_(
        self: SelfTransformIdentityComposition, output: Any, /, *args: Any, **kwargs: Any
    ) -> SelfTransformIdentityComposition:
        r"""
        Inverse output back with inplacement.

        Args
        ----
        - output
            Output from the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Simply apply identity children in order.
        identity1 = self.get_child_identity1()
        identity2 = self.get_child_identity2()
        identity2.inverse_(output)
        identity1.inverse_(output)
        return self

    def fit(
        self: SelfTransformIdentityComposition,
        input: Any,
        output: Any,
        /,
        *args: Any,
        **kwargs: Any,
    ) -> SelfTransformIdentityComposition:
        r"""
        Fit transformation parameters by example input and output.

        Args
        ----
        - input
            Example input to the transformation.
        - output
            Example output from the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def get_metadata(self: SelfTransformIdentityComposition, /) -> Mapping[str, Any]:
        r"""
        Get metadata of the transformation.

        Args
        ----

        Returns
        -------
        - metadata
            Metadata of the transformation.
        """
        # No metadata is defined.
        return {}

    def get_numeric_data(self: SelfTransformIdentityComposition, /) -> Mapping[str, NPANYS]:
        r"""
        Get numeric data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the transformation.
        """
        # No numeric data is defined.
        return {}

    def get_alphabetic_data(self: SelfTransformIdentityComposition, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # No alphabetic data is defined.
        return {}

    def set_metadata(
        self: SelfTransformIdentityComposition, metadata: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformIdentityComposition:
        r"""
        Set metadata of the transformation.

        Args
        ----
        - metadata
            Metadata of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_numeric_data(
        self: SelfTransformIdentityComposition, data: Mapping[str, NPANYS], /  # noqa: W504
    ) -> SelfTransformIdentityComposition:
        r"""
        Set numeric data of the transformation.

        Args
        ----
        - data
            Numeric data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def set_alphabetic_data(
        self: SelfTransformIdentityComposition, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransformIdentityComposition:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def get_child_identity1(
        self: SelfTransformIdentityComposition, /  # noqa: W504
    ) -> lib.transforms.TransformIdentity:
        r"""
        Get child correspond to "identity1".

        Args
        ----

        Returns
        -------
        - child
            Child transformation.
        """
        # Get child transformation.
        return cast(lib.transforms.TransformIdentity, self._children["identity1"])

    def get_child_identity2(
        self: SelfTransformIdentityComposition, /  # noqa: W504
    ) -> lib.transforms.TransformIdentity:
        r"""
        Get child correspond to "identity2".

        Args
        ----

        Returns
        -------
        - child
            Child transformation.
        """
        # Get child transformation.
        return cast(lib.transforms.TransformIdentity, self._children["identity2"])


class TransformIdentityCompositionRedundant(TransformIdentityComposition):
    r"""
    Transformation for identity composition with redundant child.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "test_identity_composition_redundant"

    def get_child_identity3(
        self: SelfTransformIdentityComposition, /  # noqa: W504
    ) -> lib.transforms.TransformIdentity:
        r"""
        Get child correspond to redundant "identity3".

        Args
        ----

        Returns
        -------
        - child
            Child transformation.
        """
        # Get child transformation.
        return cast(lib.transforms.TransformIdentity, self._children["identity3"])
