# Import Python packages.
from typing import Any, List, Mapping, Tuple

# Import external packages.
import numpy as np
import pandas as pd

# Import PyTest packagtes.
import pytest

# Import PyTest external packages.
from py._path.local import LocalPath
from sklearn.preprocessing import QuantileTransformer  # type: ignore[import-untyped]

# Import developing library.
import fin_tech_py_toolkit as lib

# Import testing library.
from ....utils import eq_dataframe, to_eq_plural_ordered
from ...utils import template_test_io, template_test_transform


# Type aliases.
Input = List[pd.DataFrame]
Output = List[pd.DataFrame]


# Runtime constants.
IDENTIFIER = lib.transforms.TransformQuantilizePandas._IDENTIFIER


def synthesize() -> Tuple[Tuple[Input, Output], Input, Output, Mapping[str, Any]]:
    r"""
    Synthesize test I/O.

    Args
    ----

    Returns
    -------
    - example
        Input and output examples.
    - input
        Input case.
    - output
        Output case.
    - supplement
        Supplementary materical for synthesized test.
    """
    # Perform PCA directly.
    n_quantiles = 2
    seed = 42
    raw = np.array(
        [
            [0.1, 0.2, -0.3, 0.1, 0.03, 0.9],
            [0.2, 0.4, -0.6, 0.9, 0.19, 0.1],
            [0.3, 0.6, -0.3, 0.5, 0.11, 0.5],
        ]
    )
    processed = QuantileTransformer(n_quantiles=n_quantiles, random_state=seed).fit_transform(raw)
    input = [pd.DataFrame(raw, columns=[str(i) for i in range(6)])]
    output = [pd.DataFrame(processed, columns=[str(i) for i in range(6)])]
    example_input: Input
    example_input = input
    example_output: Output
    example_output = []
    scikit_learn_init_kwargs = dict(n_quantiles=n_quantiles, random_state=None)
    return (
        (example_input, example_output),
        input,
        output,
        {"scikit_learn_init_kwargs": scikit_learn_init_kwargs},
    )


@pytest.mark.parametrize(
    ("raw_input", "raw_output"),
    [
        pytest.param(
            ...,
            None,
            id="unsupport-input",
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        ),
        pytest.param(
            None,
            ...,
            id="unsupport-output",
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        ),
        pytest.param(None, None, id="both-null"),
    ],
)
def test_io(*, raw_input: Any, raw_output: Any) -> None:
    r"""
    Test transformation input and output domain formalization.

    Args
    ----
    - raw_input
        Raw input.
    - raw_output
        Raw output.

    Returns
    -------
    """
    # Initialize testing transformation.
    factory = lib.transforms.FactoryTransform()

    # Run test template.
    template_test_io(
        IDENTIFIER,
        factory,
        raw_input,
        raw_output,
        to_eq_plural_ordered(eq_dataframe),
        to_eq_plural_ordered(eq_dataframe),
        fit_kwargs=dict(scikit_learn_init_kwargs=dict(n_quantiles=2, random_state=42)),
    )


def test_default(*, tmpdir: LocalPath) -> None:
    r"""
    Test transformation for PCA on Pandas data.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.transforms.FactoryTransform()

    # Generate inputs and outputs.
    example, input, output, supp = synthesize()

    # Run test template.
    template_test_transform(
        root,
        IDENTIFIER,
        factory,
        example,
        input,
        output,
        to_eq_plural_ordered(eq_dataframe),
        to_eq_plural_ordered(eq_dataframe),
        fit_kwargs=dict(scikit_learn_init_kwargs=supp["scikit_learn_init_kwargs"]),
    )


def test_empty(*, tmpdir: LocalPath) -> None:
    r"""
    Test transformation for CCA on empty Pandas data.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.transforms.FactoryTransform()

    # Generate inputs and outputs.
    transform = factory.from_args(IDENTIFIER)
    input = transform.input(None)
    output = transform.output(None)
    example: Tuple[Input, Output]
    example = (input, [])

    # Run test template.
    template_test_transform(
        root,
        IDENTIFIER,
        factory,
        example,
        input,
        output,
        to_eq_plural_ordered(eq_dataframe),
        to_eq_plural_ordered(eq_dataframe),
        fit_kwargs=dict(scikit_learn_init_kwargs=dict(n_quantiles=2, random_state=42)),
    )
