# Import Python packages.
import copy
import os
from typing import Any, Callable, Mapping, Sequence, Tuple, TypeVar, cast

# Import developing library.
import fin_tech_py_toolkit as lib


# Type variables.
Input = TypeVar("Input")
Output = TypeVar("Output")


def template_test_io(
    identifier: str,
    factory: lib.transforms.FactoryTransform,
    raw_input: Any,
    raw_output: Any,
    ineq: Callable[[Input, Input], bool],
    outeq: Callable[[Output, Output], bool],
    /,
    *,
    require_test_transform: bool = True,
    require_test_inverse: bool = True,
    init_args: Sequence[Any] = [],
    init_kwargs: Mapping[str, Any] = {},
    fit_args: Sequence[Any] = [],
    fit_kwargs: Mapping[str, Any] = {},
    transform_args: Sequence[Any] = [],
    transform_kwargs: Mapping[str, Any] = {},
    inverse_args: Sequence[Any] = [],
    inverse_kwargs: Mapping[str, Any] = {},
) -> None:
    r"""
    Template of testing input and output domain formalization.

    Args
    ----
    - identifier
        Testing transformation identifier.
    - factory
        Factory used to reproduce testing transformation instance.
    - raw_input
        Raw input data to be formalized.
    - raw_output
        Raw output data to be formalized.
    - ineq
        Equality comparator on input domain.
    - outeq
        Equality comparator on output domain.
    - require_test_transform
        Require to test transformation without inplacement.
    - require_test_inverse
        Require to test inversion without inplacement.
    - init_args
        Positional arguments for testing initialization.
    - init_kwargs
        Keyword arguments for testing initialization.
    - fit_args
        Positional arguments for testing parameter fitting.
    - fit_kwargs
        Keyword arguments for testing parameter fitting.
    - transform_args
        Positional arguments for testing transformation.
    - transform_kwargs
        Keyword arguments for testing transformation.
    - inverse_args
        Positional arguments for testing inversion.
    - inverse_kwargs
        Keyword arguments for testing inversion.

    Returns
    -------
    """
    # Finalize runtime flags.
    transform = cast(
        lib.transforms.BaseTransform[Input, Output],
        factory.from_args(identifier, *init_args, **init_kwargs),
    )
    tags = transform.tags
    test_transform = require_test_transform
    test_inverse = require_test_inverse and tags.invertible

    # Formalize input, output and parameters.
    input = transform.input(raw_input)
    output = transform.output(raw_output)
    if tags.parametric:
        # Run parameter fitting.
        transform.fit(input, output, *fit_args, **fit_kwargs)

    # Run test snippets based on flags.
    if test_transform:
        # Run transformation without inplacement.
        input_ = copy.deepcopy(input)
        output_ = transform.transform(input_, *transform_args, **transform_kwargs)
        assert outeq(output_, output)

    # Run test snippets based on flags.
    if test_inverse:
        # Run inversion without inplacement.
        output_ = copy.deepcopy(output)
        input_ = transform.inverse(output_, *inverse_args, **inverse_kwargs)
        assert ineq(input_, input)


def template_test_transform(
    root: str,
    identifier: str,
    factory: lib.transforms.FactoryTransform,
    example: Tuple[Input, Output],
    input: Input,
    output: Output,
    ineq: Callable[[Input, Input], bool],
    outeq: Callable[[Output, Output], bool],
    /,
    *,
    require_test_fit: bool = True,
    require_test_transform: bool = True,
    require_test_transform_: bool = True,
    require_test_inverse: bool = True,
    require_test_inverse_: bool = True,
    init_args: Sequence[Any] = [],
    init_kwargs: Mapping[str, Any] = {},
    fit_args: Sequence[Any] = [],
    fit_kwargs: Mapping[str, Any] = {},
    transform_args: Sequence[Any] = [],
    transform_kwargs: Mapping[str, Any] = {},
    inverse_args: Sequence[Any] = [],
    inverse_kwargs: Mapping[str, Any] = {},
) -> None:
    r"""
    Template of testing transformation.

    Args
    ----
    - root
        File system storage root.
    - identifier
        Testing transformation identifier.
    - factory
        Factory used to reproduce testing transformation instance.
    - example
        Testing input and output examples for parameter fitting.
    - input
        Testing input case.
    - output
        Testing output case.
    - ineq
        Equality comparator on input domain.
    - outeq
        Equality comparator on output domain.
    - require_test_fit
        Require to test parameter fitting.
    - require_test_transform
        Require to test transformation without inplacement.
    - require_test_transform_
        Require to test transformation with inplacement.
    - require_test_inverse
        Require to test inversion without inplacement.
    - require_test_inverse_
        Require to test inversion with inplacement.
    - init_args
        Positional arguments for testing initialization.
    - init_kwargs
        Keyword arguments for testing initialization.
    - fit_args
        Positional arguments for testing parameter fitting.
    - fit_kwargs
        Keyword arguments for testing parameter fitting.
    - transform_args
        Positional arguments for testing transformation.
    - transform_kwargs
        Keyword arguments for testing transformation.
    - inverse_args
        Positional arguments for testing inversion.
    - inverse_kwargs
        Keyword arguments for testing inversion.

    Returns
    -------
    """
    # Finalize runtime flags.
    transform = cast(
        lib.transforms.BaseTransform[Input, Output],
        factory.from_args(identifier, *init_args, **init_kwargs),
    )
    tags = transform.tags
    test_fit = require_test_fit and tags.parametric
    test_transform = require_test_transform
    test_transform_ = require_test_transform and tags.inplacable
    test_inverse = require_test_inverse and tags.invertible
    test_inverse_ = require_test_inverse_ and tags.invertible and tags.inplacable

    # Reserve essential temporary directories.
    params = os.path.join(root, "parameters")
    assert not os.path.isdir(params)

    # Run test snippets based on flags.
    if test_fit:
        # Run parameter fitting.
        transform.fit(*example, *fit_args, **fit_kwargs)
    transform.save(params)
    transform = factory.from_filesys(params)

    # Run test snippets based on flags.
    if test_transform:
        # Run transformation without inplacement.
        input_ = copy.deepcopy(input)
        output_ = transform.transform(input_, *transform_args, **transform_kwargs)
        assert outeq(output_, output)

    # Run test snippets based on flags.
    if test_transform_:
        # Run transformation without inplacement.
        input_ = copy.deepcopy(input)
        transform.transform_(input_, *transform_args, **transform_kwargs)
        assert ineq(input_, cast(Input, output))

    # Run test snippets based on flags.
    if test_inverse:
        # Run inversion without inplacement.
        output_ = copy.deepcopy(output)
        input_ = transform.inverse(output_, *inverse_args, **inverse_kwargs)
        assert ineq(input_, input)

    # Run test snippets based on flags.
    if test_inverse_:
        # Run inversion with inplacement.
        output_ = copy.deepcopy(output)
        transform.inverse_(output_, *inverse_args, **inverse_kwargs)
        assert outeq(output_, cast(Output, input))

    # Run test snippets based on flags.
    if test_fit and test_transform:
        # Run transformation without inplacement with parameter fitting.
        input_ = copy.deepcopy(input)
        output_ = transform.fit_transform(
            example, input_, *fit_args, *transform_args, **fit_kwargs, **transform_kwargs
        )
        assert outeq(output_, output)

    # Run test snippets based on flags.
    if test_fit and test_transform_:
        # Run transformation without inplacement with parameter fitting.
        input_ = copy.deepcopy(input)
        transform.fit_transform_(
            example, input_, *fit_args, *transform_args, **fit_kwargs, **transform_kwargs
        )
        assert ineq(input_, cast(Input, output))

    # Run test snippets based on flags.
    if test_fit and test_inverse:
        # Run inversion without inplacement with parameter fitting.
        output_ = copy.deepcopy(output)
        input_ = transform.fit_inverse(
            example, output_, *fit_args, *inverse_args, **fit_kwargs, **inverse_kwargs
        )
        assert ineq(input_, input)

    # Run test snippets based on flags.
    if test_fit and test_inverse_:
        # Run inversion with inplacement with parameter fitting.
        output_ = copy.deepcopy(output)
        transform.fit_inverse_(
            example, output_, *fit_args, *inverse_args, **fit_kwargs, **inverse_kwargs
        )
        assert outeq(output_, cast(Output, input))
