# Import Python packages.
from typing import Any, Dict, List, Mapping, Sequence, Tuple, cast

# Import developing packages.
import fin_tech_py_toolkit as lib


# Runtime constants.
MAX_DIMS = 128


def prepare(
    name: str, prefix: str, suffix: str, /  # noqa: W504
) -> Tuple[lib.transforms.BaseTransformPandas, Sequence[Any], Mapping[str, Any]]:
    r"""
    Prepare transformation.

    Args
    ----
    - name
        Transformation name.
    - prefix
        Cache prefix.
    - suffix
        Cache suffix.

    Returns
    -------
    - transform
        A transformation.
    - fit_args
        Positional arguments for parameter fitting.
    - fit_kwargs
        Keyword arguments for parameter fitting.
    """
    # Prepare preprecoessing.
    transform: lib.transforms.BaseTransformPandas
    fit_args: List[Any]
    fit_kwargs: Dict[str, Any]
    if name == "pca":
        # PCA dimensionality reduction.
        transform = lib.transforms.TransformPCAPandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(scikit_learn_init_kwargs=dict(n_components=MAX_DIMS, random_state=42))
    elif name == "featagglo":
        # Feature Agglomeration dimensionality reduction.
        transform = lib.transforms.TransformFeatAggloPandas(
            cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
        )
        fit_args = []
        fit_kwargs = dict(scikit_learn_init_kwargs=dict(n_clusters=MAX_DIMS, pooling_func="mean"))
    elif name == "identity":
        # Identity dimensionality reduction.
        transform = cast(
            lib.transforms.BaseTransformPandas,
            lib.transforms.TransformIdentity(
                cache_prefix=prefix, cache_suffix=suffix, allow_alias=True
            ),
        )
        fit_args = []
        fit_kwargs = dict()
    else:
        # Unsupporting transformation.
        raise NotImplementedError
    return transform, fit_args, fit_kwargs
