# Import Python packages.
import abc
import functools
import re
from typing import Any, Callable, Dict, Generic, Mapping, NamedTuple, Type, TypeVar, cast

# Import relatively from other modules.
from ..types import NPANYS


# Type variables.
Content = TypeVar("Content")


# Self types.
SelfTypeData = TypeVar("SelfTypeData", bound="TypeData")
SelfBaseData = TypeVar("SelfBaseData", bound="BaseData[Any]")


class ErrorDataUnsupport(Exception):
    r"""
    Exception for unsupported functionality of data.
    Pay attention that this should not be used for partially unsupported functionality with
    conditions.
    """


class ErrorDataUnsupportPartial(Exception):
    r"""
    Exception for partially unsupported functionality of data.
    Raise this if functionality is defined, but is conditionally unsupported due to configurations,
    arguments, accessibilities or other factors.
    """


class DataTags(NamedTuple):
    r"""
    Tags of the data.
    """
    # Annotations.
    emptiable: bool
    ambiguous: bool


# Create datum class registration container.
_REGISTRATIONS: Dict[str, Type["BaseData[Any]"]]
_REGISTRATIONS = {}


class TypeData(abc.ABCMeta):
    r"""
    Metaclass of data.
    """

    @staticmethod
    def __new__(cls: Type[SelfTypeData], /, *args: Any, **kwargs: Any) -> Type["BaseData[Any]"]:
        r"""
        Create a new instance of the class.

        Args
        ----
        - cls
            The class (type) of creating new instance.
            The new instance is first created as generic (any) type, then is casted into the class
            (type).

        Returns
        -------
        - obj
            A new instance casted by the class (type).
        """
        # Create a class by super call.
        data = cast(Type["BaseData[Any]"], abc.ABCMeta.__new__(cls, *args, **kwargs))

        # Register the transformation class.
        global _REGISTRATIONS
        identifier = data._IDENTIFIER
        assert (
            re.fullmatch(r"[a-zA-Z0-9_]+(\.[a-zA-Z0-9_]+)*", identifier) is not None
        ), f'Registering data identifier: "{identifier:s}" is in wrong format.'
        assert (
            identifier not in _REGISTRATIONS
        ), f'Registering data identifier: "{identifier:s}" is already registered.'
        _REGISTRATIONS[identifier] = data
        return data


class BaseData(Generic[Content], abc.ABC, metaclass=TypeData):
    r"""
    Base of data container.
    """
    # Annotations at class level.
    _IDENTIFIER: str

    # Transformation unique identifier.
    _IDENTIFIER = "_base"

    def __init__(
        self: SelfBaseData,
        content: Content,
        /,
        *args: Any,
        allow_alias_disambiguition: bool = False,
        **kwargs: Any,
    ) -> None:
        r"""
        Initialize the class.

        Args
        ----
        - content
            Content in the datum.
        - allow_alias
            If True, Allow disambiguous content to be an alias of other object.
            Otherwise, the class will make a completed copy of the content during disambiguition.
            Since most disambiguition will modify the content, aliasing is forbidden by default.

        Returns
        -------
        """
        # Save essential attributes.
        self.allow_alias_disambiguition = allow_alias_disambiguition

        # Annotate essential attributes.
        self._content: Content

    @classmethod
    def _is_defined(
        cls: Type[SelfBaseData], f: Callable[..., Any], /, *args: Any, **kwargs: Any
    ) -> bool:
        r"""
        Get definition status by calling method with given arguments.

        Args
        ----

        Returns
        -------
        - flag
            Definition status.
        """
        # Collect implementation flag.
        try:
            # Call the method.
            f(*args, **kwargs)
        except ErrorDataUnsupport:
            # The method is not defined.
            return False
        else:
            # The method is defined.
            return True

    @functools.cached_property
    def _is_defined_identity(self: SelfBaseData, /) -> bool:
        r"""
        Get definition status of identity generation

        Args
        ----

        Returns
        -------
        - flag
            Definition status.
        """
        # Collect implementation flag of identity generation.
        return self._is_defined(self.identity)

    @functools.cached_property
    def _is_defined__disambiguate(self: SelfBaseData, /) -> bool:
        r"""
        Get definition status of disambiguation with inplacement.

        Args
        ----

        Returns
        -------
        - flag
            Definition status.
        """
        # Collect implementation flag of disambiguation with inplacement.
        return self._is_defined(self._disambiguate, deep=not self.allow_alias_disambiguition)

    @property
    def tags(self: SelfBaseData, /) -> DataTags:
        r"""
        Tags of the transformation.

        Args
        ----

        Returns
        -------
        - tags
            Tags of the transformation.
        """
        # Summarize flags.
        emptiable = self._is_defined_identity
        ambiguous = not self._is_defined__disambiguate

        # Construct final flag container.
        return DataTags(emptiable, ambiguous)

    def identity(self: SelfBaseData, /) -> SelfBaseData:
        r"""
        Generate identity (empty) element of the datum.

        Args
        ----

        Returns
        -------
        - obj
            Identity (empty) instance of the datum.
        """
        # By default, no operation is valid.
        raise ErrorDataUnsupport(
            f'Identity generation is not defined for "{self._IDENTIFIER:s}" data.'
        )

    def _disambiguate(self: SelfBaseData, /, *, deep: bool = True) -> SelfBaseData:
        r"""
        Remove ambiguation caused by representing or storage differences of the same datum.

        Args
        ----
        - deep
            Make a completed copy after disambiguition.

        Returns
        -------
        """
        # By default, no operation is valid.
        raise ErrorDataUnsupport(f'Disambiguation is not defined for "{self._IDENTIFIER:s}" data.')

    @abc.abstractmethod
    def save(self: SelfBaseData, path: str, /) -> SelfBaseData:
        r"""
        Save the content to file system.

        Args
        ----
        - path
            Path to save the content.

        Returns
        -------
        - self
            Class instance itself.
        """

    @abc.abstractmethod
    def load(self: SelfBaseData, path: str, /) -> SelfBaseData:
        r"""
        Load the content from file system.

        Args
        ----
        - path
            Path to load the content.

        Returns
        -------
        - self
            Class instance itself.
        """

    @property
    @abc.abstractmethod
    def hashcode(self: SelfBaseData, /) -> str:
        r"""
        Collect hash code of data content.

        Args
        ----

        Returns
        -------
        - code
            Hash code.
        """

    @abc.abstractmethod
    def to_numeric(self: SelfBaseData, /, *args: Any, **kwargs: Any) -> Mapping[str, NPANYS]:
        r"""
        Translate into numeric format.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the content.
            Numeric data include booleans, integers, floatings, strings, lists, mappings and other
            serializable formats.
        """

    @classmethod
    @abc.abstractmethod
    def from_numeric(
        cls: Type[SelfBaseData], data: Mapping[str, NPANYS], /, *args: Any, **kwargs: Any
    ) -> SelfBaseData:
        r"""
        Translate from numeric format.

        Args
        ----
        - data
            Numeric data of the content.
            Numeric data include booleans, integers, floatings, strings, lists, mappings and other
            serializable formats.

        Returns
        -------
        """
