# Import Python packages.
import os
import shutil
from typing import Optional

# Import relatively from other modules.
from ..time import wait_until_true


def rmtree(
    path: str, /, *, timeout: Optional[float] = None, interval: Optional[float] = None
) -> None:
    r"""
    Remove a directory thoroughly.

    Args
    ----
    - path
        Directory path to be removed.
    - timeout
        Maximum seconds to wait before raising timeout error.
    - interval
        The interval in seconds to check removal status.

    Returns
    -------
    """
    # Call Python implementation.
    # In rare cases, e.g., deep hierarchy or windows system, directory may not be fully deleted due
    # to lazy deletion, so waiting is necessary.
    shutil.rmtree(path)
    wait_until_true(os.path.isdir, path, negate=True, timeout=None, interval=None)


def rmfile(
    path: str, /, *, timeout: Optional[float] = None, interval: Optional[float] = None
) -> None:
    r"""
    Remove a file thoroughly.

    Args
    ----
    - path
        File path to be removed.
    - timeout
        Maximum seconds to wait before raising timeout error.
    - interval
        The interval in seconds to check removal status.

    Returns
    -------
    """
    # Call Python implementation.
    # In rare cases, e.g., deep hierarchy or windows system, file may not be fully deleted due to
    # lazy deletion, so waiting is necessary.
    os.remove(path)
    wait_until_true(os.path.isfile, path, negate=True, timeout=None, interval=None)


def mkdirs(
    path: str,
    /,
    *,
    must_be_new: bool = False,
    allow_auto_remove: bool = False,
    timeout: Optional[float] = None,
    interval: Optional[float] = None,
) -> None:
    r"""
    Make a directory.

    Args
    ----
    - path
        Directory path to be created.
    - must_be_new
        If True, the directory must be newly made, thus removal is required if it already exists.
        Otherwise, it will do nothing if the directory already exists.
    - allow_auto_remove
        If True, automatic removal is allowed before directory making.
        Otherwise, removal must be explicitly called before calling this function.
    - timeout
        Maximum seconds to wait before raising timeout error.
    - interval
        The interval in seconds to check removal status.

    Returns
    -------
    """
    # Collect essential flags.
    is_existing = os.path.isdir(path)

    # Only when directory already exists, emptyness is require and removal is allowed, the function
    # can silently remove existing directory by its own.
    if is_existing and must_be_new and allow_auto_remove:
        # Thoroughly remove directory before making.
        rmtree(path, timeout=timeout, interval=interval)

    # Call Python implementation.
    # In rare cases, e.g., deep hierarchy or windows system, directory may not be completely made
    # due to lazy making, so waiting is necessary.
    os.makedirs(path, exist_ok=not must_be_new)
    wait_until_true(os.path.isdir, path, timeout=timeout, interval=interval)
