# Import Python packages.
from typing import Any, Mapping, TypeVar

# Import relatively from other modules.
from ...datasets import BaseDataset
from ...transforms import BaseTransformList
from ...types import NPANYS


# Type variables.
Input = TypeVar("Input", bound="BaseDataset[Any]")
Output = TypeVar("Output")


# Self types.
SelfTransdatasetUnravel = TypeVar("SelfTransdatasetUnravel", bound="TransdatasetUnravel[Any, Any]")


class TransdatasetUnravel(BaseTransformList[Input, Output]):
    r"""
    Transformation for unraveling dataset memory data into generic transformation memory data.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "unravel.dataset"

    def __init__(
        self: SelfTransdatasetUnravel, /, *args: Any, allow_alias: bool = True, **kwargs: Any
    ) -> None:
        r"""
        Initialize the class.

        Args
        ----
        - allow_alias
            If True, Allow content is an alias of other object.
            Otherwise, the class will make a completed copy of the content.

        Returns
        -------
        """
        # Super call.
        BaseTransformList.__init__(self, *args, **kwargs)

        # Save essential arguments.
        self.allow_alias = allow_alias

    def get_metadata(self: SelfTransdatasetUnravel, /) -> Mapping[str, Any]:
        r"""
        Get metadata of the transformation.

        Args
        ----

        Returns
        -------
        - metadata
            Metadata of the transformation.
        """
        # Collect essential attributes.
        return {"allow_alias": self.allow_alias}

    def get_numeric_data(self: SelfTransdatasetUnravel, /) -> Mapping[str, NPANYS]:
        r"""
        Get numeric data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the transformation.
        """
        # Do nothing.
        return {}

    def set_metadata(
        self: SelfTransdatasetUnravel, metadata: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransdatasetUnravel:
        r"""
        Set metadata of the transformation.

        Args
        ----
        - metadata
            Metadata of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Safety check.
        assert "allow_alias" in metadata, "Data container alias flag is missing"

        # Get data container alias flag.
        self.allow_alias = metadata["allow_alias"]
        return self

    def set_numeric_data(
        self: SelfTransdatasetUnravel, data: Mapping[str, NPANYS], /  # noqa: W504
    ) -> SelfTransdatasetUnravel:
        r"""
        Set numeric data of the transformation.

        Args
        ----
        - data
            Numeric data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self
