# Import Python packages.
import warnings
from typing import Any, Sequence, TypeVar

# Import external packages.
import numpy as np


# Type variables.
Series = TypeVar("Series", bound="Sequence[Any]")


def nancount(series: Sequence[Any], /) -> int:
    r"""
    Get count of null elements.

    Args
    ----
    - series
        A series for collection.

    Returns
    -------
    - value
        Count of null elements.
    """
    # Count hits of null value.
    return int(np.sum(np.isnan(series)))


def nanratio(series: Sequence[Any], /) -> float:
    r"""
    Get ratio of null elements.

    Args
    ----
    - series
        A series for collection.

    Returns
    -------
    - value
        Ratio of null elements.
    """
    # Count hits of null value.
    return float(np.sum(np.isnan(series))) / float(len(series))


def nandeg(series: Sequence[Any], /) -> int:
    r"""
    Get degree of valued elements.

    Args
    ----
    - series
        A series for collection.

    Returns
    -------
    - value
        Degree of valued elements.
    """
    # Exclude count of null values from total length.
    return len(series) - nancount(series)


def nanmin(series: Sequence[Any], /, null: float = float("nan")) -> float:
    r"""
    Get minimum of valued elements.

    Args
    ----
    - series
        A series for collection.
    - null
        Default value when no valued element is presented.

    Returns
    -------
    - value
        Minimum of valued elements.
    """
    # Get the minimum with autofilling for totally null case.
    with warnings.catch_warnings():
        # Warning w.r.t totally null case should be silent.
        warnings.filterwarnings(
            "ignore", message="All-NaN axis encountered", category=RuntimeWarning
        )
        value = float(np.nanmin(series))
    value = null if np.isnan(value) else value
    return value


def nanmax(series: Sequence[Any], /, null: float = float("nan")) -> float:
    r"""
    Get maximum of valued elements.

    Args
    ----
    - series
        A series for collection.
    - null
        Default value when no valued element is presented.

    Returns
    -------
    - value
        Maximum of valued elements.
    """
    # Get the maximum with autofilling for totally null case.
    with warnings.catch_warnings():
        # Warning w.r.t totally null case should be silent.
        warnings.filterwarnings(
            "ignore", message="All-NaN axis encountered", category=RuntimeWarning
        )
        value = float(np.nanmax(series))
    value = null if np.isnan(value) else value
    return value


def nanmean(series: Sequence[Any], /, null: float = float("nan")) -> float:
    r"""
    Get mean of valued elements.

    Args
    ----
    - series
        A series for collection.
    - null
        Default value when no valued element is presented.

    Returns
    -------
    - value
        Mean of valued elements.
    """
    # Get the mean with autofilling for totally null case.
    with warnings.catch_warnings():
        # Warning w.r.t totally null case should be silent.
        warnings.filterwarnings("ignore", message="Mean of empty slice", category=RuntimeWarning)
        value = float(np.nanmean(series))
    value = null if np.isnan(value) else value
    return value


def nanstd(series: Sequence[Any], /, null: float = float("nan"), ddof: int = 0) -> float:
    r"""
    Get standard deviation of valued elements.

    Args
    ----
    - series
        A series for collection.
    - null
        Default value when no valued element is presented.
    - ddof
        Delta degree of freedom.

    Returns
    -------
    - value
        Maximum of valued elements.
    """
    # Get the standard deviation with autofilling for totally null case.
    length = nandeg(series)
    with warnings.catch_warnings():
        # Warning w.r.t totally null case should be silent.
        warnings.filterwarnings(
            "ignore", message="Degrees of freedom <= 0 for slice.", category=RuntimeWarning
        )
        value = float(np.nanstd(series, ddof=min(ddof, length - 1)))
    value = null if np.isnan(value) else value
    return value


def normdeg_identity(series: Series, /) -> Series:
    r"""
    Normalize degree column by identity.

    Args
    ----
    - series
        Degree column.

    Returns
    -------
    - series
        Normalized degree column.
    """
    # Do nothing.
    return series


def normdeg_log(series: Series, /) -> Series:
    r"""
    Normalize degree column by logarithm.

    Args
    ----
    - series
        Degree column.

    Returns
    -------
    - series
        Normalized degree column.
    """
    # Plus 1 to ensure safe logarithm.
    series = np.log(np.add(series, 1))
    return series


def normdeg_log_normal(series: Series, /) -> Series:
    r"""
    Normalize degree column by logarithm normalization.

    Args
    ----
    - series
        Degree column.

    Returns
    -------
    - series
        Normalized degree column.
    """
    # Apply safe min-max normalization after safe logarithm.
    series = normdeg_log(series)
    series = series - np.min(series)
    series = series / (np.max(series) if np.max(series) > 0.0 else 1.0)
    return series
