# Import Python packages.
import copy
import hashlib
import json
from typing import Any, Mapping, Sequence, Type, TypeVar

# Import external packages.
import numpy as np

# Import developing library.
import fin_tech_py_toolkit as lib
from fin_tech_py_toolkit.types import NPANYS


# Self types.
SelfDataNull = TypeVar("SelfDataNull", bound="DataNull")
SelfDataInts = TypeVar("SelfDataInts", bound="DataInts")


class DataNull(lib.data.BaseData[None]):
    r"""
    Data container of nothing.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "test_null"

    def save(self: SelfDataNull, path: str, /) -> SelfDataNull:
        r"""
        Save the content to file system.

        Args
        ----
        - path
            Path to save the content.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    def load(self: SelfDataNull, path: str, /) -> SelfDataNull:
        r"""
        Load the content from file system.

        Args
        ----
        - path
            Path to load the content.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self

    @property
    def hashcode(self: SelfDataNull, /) -> str:
        r"""
        Collect hash code of data content.

        Args
        ----

        Returns
        -------
        - code
            Hash code.
        """
        # No data results in empty hash code.
        return ""

    def to_numeric(self: SelfDataNull, /) -> Mapping[str, NPANYS]:
        r"""
        Translate into numeric format.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the content.
            Numeric data include booleans, integers, floatings, strings, lists, mappings and other
            serializable formats.
        """
        # No data has no numeric data.
        return {}

    @classmethod
    def from_numeric(
        cls: Type[SelfDataNull], data: Mapping[str, NPANYS], /, *args: Any, **kwargs: Any
    ) -> SelfDataNull:
        r"""
        Translate from numeric format.

        Args
        ----
        - data
            Numeric data of the content.
            Numeric data include booleans, integers, floatings, strings, lists, mappings and other
            serializable formats.

        Returns
        -------
        """
        # Convert numeric data back to the content, and create the instance.
        return cls(*args, **kwargs)


class DataInts(lib.data.BaseData[Sequence[int]]):
    r"""
    Data container of integers.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "test_ints"

    def __init__(self: SelfDataInts, content: Sequence[int], /, *arg: Any, **kwargs: Any) -> None:
        r"""
        Initialize the class.

        Args
        ----
        - content
            Content in the datum.

        Returns
        -------
        """
        # Super call.
        lib.data.BaseData.__init__(self, content, *arg, **kwargs)

        # Disambiguite the content.
        self._content = content
        self._disambiguate(deep=not self.allow_alias_disambiguition)

    def identity(self: SelfDataInts, /) -> SelfDataInts:
        r"""
        Generate identity (empty) element of the datum.

        Args
        ----

        Returns
        -------
        - obj
            Identity (empty) instance of the datum.
        """
        # This is a list of integers, so identity element is an empty list.
        return self.__class__([])

    def _disambiguate(self: SelfDataInts, /, *, deep: bool = True) -> SelfDataInts:
        r"""
        Remove ambiguation caused by representing or storage differences of the same datum.

        Args
        ----
        - deep
            Make a completed copy after disambiguition.

        Returns
        -------
        """
        # As a list, the original order matters, thus do nothing.
        if deep:
            # If the content is not an alias, make a completed copy.
            self._content = copy.deepcopy(self._content)
        return self

    def save(self: SelfDataInts, path: str, /) -> SelfDataInts:
        r"""
        Save the content to file system.

        Args
        ----
        - path
            Path to save the content.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Directly save the content.
        with open(path, "w") as file:
            # Save in readable JSON format.
            json.dump(self._content, file, indent=4)
        return self

    def load(self: SelfDataInts, path: str, /) -> SelfDataInts:
        r"""
        Load the content from file system.

        Args
        ----
        - path
            Path to load the content.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Directly load the content.
        with open(path, "r") as file:
            # Load from JSON format.
            self._content = json.load(file)
        return self

    @property
    def hashcode(self: SelfDataInts, /) -> str:
        r"""
        Collect hash code of data content.

        Args
        ----

        Returns
        -------
        - code
            Hash code.
        """
        # Directly compute the hash code of integer bytes.
        hashcoder = hashlib.sha256()
        hashcoder.update(bytes(" ".join(str(value) for value in self._content), "utf-8"))
        return hashcoder.hexdigest()

    def to_numeric(self: SelfDataInts, /) -> Mapping[str, NPANYS]:
        r"""
        Translate into numeric format.

        Args
        ----

        Returns
        -------
        - data
            Numeric data of the content.
            Numeric data include booleans, integers, floatings, strings, lists, mappings and other
            serializable formats.
        """
        # Convert data into numeric format.
        return {"content": np.array(self._content)}

    @classmethod
    def from_numeric(
        cls: Type[SelfDataInts], data: Mapping[str, NPANYS], /, *args: Any, **kwargs: Any
    ) -> SelfDataInts:
        r"""
        Translate from numeric format.

        Args
        ----
        - data
            Numeric data of the content.
            Numeric data include booleans, integers, floatings, strings, lists, mappings and other
            serializable formats.

        Returns
        -------
        """
        # Convert numeric data back to the content, and create the instance.
        return cls(data["content"].tolist(), *args, **kwargs)
