# Import Python packages.
import os
from typing import Optional, cast

# Import external packages.
import pandas as pd

# Import PyTest packagtes.
import pytest

# Import PyTest external packages.
from py._path.local import LocalPath

# Import developing library.
import fin_tech_py_toolkit as lib

# Import testing library.
from ..utils import eq_dataframe
from .utils import template_test_data


# Type aliases.
Content = pd.DataFrame


# Runtime constants.
IDENTIFIER = lib.data.DataTabular._IDENTIFIER


def synthesize(*, irregular: bool) -> Content:
    r"""
    Synthesize test I/O.

    Args
    ----
    - irregular
        If True, synthesize irregular cell values.

    Returns
    -------
    - content
        Synthetic content of arbitrary data container.
    """
    # Generate regular cell values.
    dataframe = pd.DataFrame(
        {
            "bool": [False, True, float("nan")],
            "int": [0, 1, float("nan")],
            "float": [0.0, 1.0, float("nan")],
            "str": ["No", "Yes", float("nan")],
        }
    )

    # Insert irregular data.
    if irregular:
        # Content like list, dict is irregular.
        dataframe["irregular"] = [[0, 1], {"No": 0, "Yes": 1}, float("nan")]
    return dataframe


def modify_default(dataframe: pd.DataFrame, /) -> None:
    r"""
    Modify dataframe with inplacement.

    Args
    ----
    - dataframe
        The dataframe to be modified.

    Returns
    -------
    """
    # Change some values.
    dataframe.iloc[0, 0] = False
    dataframe.iloc[0, 1] = True


def modify_empty(dataframe: pd.DataFrame, /) -> None:
    r"""
    Modify dataframe with inplacement.

    Args
    ----
    - dataframe
        The dataframe to be modified.

    Returns
    -------
    """
    # Do nothing.
    pass


@pytest.mark.parametrize(
    ("irregular", "allow_alias"),
    [
        pytest.param(
            True, True, id="irregular-alias", marks=[pytest.mark.xfail(raises=AssertionError)]
        ),
        pytest.param(
            True, False, id="irregular-copy", marks=[pytest.mark.xfail(raises=AssertionError)]
        ),
        pytest.param(
            False,
            True,
            id="regular-alias",
            marks=[pytest.mark.xfail(raises=lib.data.ErrorDataUnsupportPartial)],
        ),
        pytest.param(False, False, id="regular-copy"),
    ],
)
def test_default(*, tmpdir: LocalPath, irregular: bool, allow_alias: bool) -> None:
    r"""
    Test transformation for tabularization.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.
    - irregular
        If True, synthesize irregular cell values.
    - allow_alias
        If True, the content of data container is an alias of source content.
        If False, the content of data container is a completed copy of source content.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.data.FactoryData()

    # Generate content.
    content = synthesize(irregular=irregular)

    # Run test template.
    template_test_data(
        root,
        IDENTIFIER,
        factory,
        content,
        eq_dataframe,
        modify_default,
        allow_alias=allow_alias,
        init_kwargs=dict(sort_columns="alphabetic", sort_rows="rankable"),
    )


def test_empty(*, tmpdir: LocalPath) -> None:
    r"""
    Test transformation for tabularization on empty Pandas data.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.data.FactoryData()

    # Generate identity content.
    identity = cast(
        lib.data.DataTabular,
        factory.from_args(
            IDENTIFIER, synthesize(irregular=False), sort_columns="alphabetic", sort_rows="rankable"
        ),
    ).identity()

    # Run test template.
    template_test_data(
        root,
        IDENTIFIER,
        factory,
        identity._content,
        eq_dataframe,
        modify_empty,
        allow_alias=False,
        init_kwargs=dict(sort_columns="identity", sort_rows="identity"),
        potential_translates=[("csv", ([os.path.join(root, "csv.csv")], {}), ([], {}))],
    )


def sort_force_error(dataframe: pd.DataFrame, /) -> pd.DataFrame:
    r"""
    Sorting algorithm that always fail.

    Args
    ----
    - dataframe
        Dataframe to be sorted.

    Returns
    -------
    - dataframe
        Sorted dataframe.
    """
    # Force error directly.
    raise RuntimeError("Force error.")


@pytest.mark.xfail(raises=RuntimeError)
def test_register_sort_invalid() -> None:
    r"""
    Test registrating an invalid sorting algorithm.

    Args
    ----

    Returns
    -------
    """
    # Initialize testing transformation.
    factory = lib.data.FactoryData()

    # Generate identity content.
    identity = cast(
        lib.data.DataTabular,
        factory.from_args(
            IDENTIFIER, synthesize(irregular=False), sort_columns="alphabetic", sort_rows="rankable"
        ),
    ).identity()

    # Register invalid sorting algorithm.
    identity.register_sort(sort_force_error, "columns", "test_invalid")
    identity.register_sort(sort_force_error, "rows", "test_invalid")
    identity.register_sort(sort_force_error, "test_both", "test_invalid_init")
    identity.register_sort(sort_force_error, "test_both", "test_invalid_add")

    # Generate identity content but use invalid sorting.
    factory.from_args(
        IDENTIFIER, identity._content, sort_columns="test_invalid", sort_rows="test_invalid"
    )


@pytest.mark.parametrize(
    ("sort_columns", "sort_rows"),
    [
        pytest.param(
            None,
            "identity",
            id="columns",
            marks=[pytest.mark.xfail(raises=lib.data.ErrorDataUnsupportPartial)],
        ),
        pytest.param(
            "identity",
            None,
            id="rows",
            marks=[pytest.mark.xfail(raises=lib.data.ErrorDataUnsupportPartial)],
        ),
    ],
)
def test_sort(*, sort_columns: Optional[str], sort_rows: Optional[str]) -> None:
    r"""
    Test all sorting algorithms.

    Args
    ----
    - sort_columns
        Column sorting algorithm name.
    - sort_rows
        Row sorting algorithm name.

    Returns
    -------
    """
    # Initialize testing transformation.
    factory = lib.data.FactoryData()

    # Generate identity content.
    factory.from_args(
        IDENTIFIER, synthesize(irregular=False), sort_columns=sort_columns, sort_rows=sort_rows
    )
