# Import Python packages.
import json
import os
from typing import Any, Optional, Sequence, Tuple

# Import PyTest packagtes.
import pytest

# Import PyTest external packages.
from py._path.local import LocalPath

# Import developing library.
import fin_tech_py_toolkit as lib

# Import testing library.
from .supp_base import DatasetInts


# Runtime constants.
IDENTIFIER = DatasetInts._IDENTIFIER


@pytest.mark.parametrize(
    ("raw_input", "raw_output"),
    [
        pytest.param(
            ...,
            None,
            id="unsupport-input",
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        ),
        pytest.param(
            None,
            ...,
            id="unsupport-output",
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        ),
        pytest.param(None, None, id="both-null"),
    ],
)
def test_io(*, raw_input: Any, raw_output: Any) -> None:
    r"""
    Test dataset loading transformation input and output domain formalization.

    Args
    ----
    - raw_input
        Raw input.
    - raw_output
        Raw output.

    Returns
    -------
    """
    # Create a dataset and perform formalization.
    dataset = DatasetInts()
    dataset.input(raw_input)
    dataset.output(raw_output)


def test_ints(*, tmpdir: LocalPath) -> None:
    r"""
    Test dataset of integers.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Create dataset source files.
    root = str(tmpdir)
    source = [1, 2, 3]
    lib.io.mkdirs(os.path.join(root, "raw"))
    with open(os.path.join(root, "raw", "source.json"), "w") as file:
        # Save in reedable JSON format.
        json.dump(source, file, indent=4)
    addresses: Sequence[Tuple[str, Optional[str]]]
    addresses = [("source", os.path.join(root, "raw", "source.json"))]

    # Load dataset from file system.
    dataset = DatasetInts.from_storage(addresses)

    # Dataset tags should be fixed.
    tags = dataset.tags
    assert not tags.inplacable
    assert not tags.invertible
    assert tags.parametric

    # Transformation of dataset indeed is indexing data blocks.
    dataset.transform([])
    (source_,) = dataset.transform(addresses)
    assert tuple(source_._content) == tuple(source)

    # Parameter fitting of the dataset indeed is loading data blocks into memory.
    dataset_ = DatasetInts()
    dataset_.fit(addresses, [])
    (source_,) = dataset.transform([(name, None) for name, _ in addresses])
    assert tuple(source_._content) == tuple(source)
