# Import Python packages.
from typing import Any, List, Mapping, TypeVar

# Import external packages.
import numpy as np

# Import developing library.
import fin_tech_py_toolkit as lib
from fin_tech_py_toolkit.types import NPINTS

# Import testing library.
from ...data.supp_base import DataInts
from ...datasets.supp_base import DatasetInts


# Type aliases.
Input = List[DatasetInts]
Output = List[NPINTS]


# Self types.
SelfTransdatasetUnravelInts = TypeVar(
    "SelfTransdatasetUnravelInts", bound="TransdatasetUnravelInts"
)


class TransdatasetUnravelInts(lib.transdatasets.TransdatasetUnravel[DatasetInts, NPINTS]):
    r"""
    Transformation for unraveling integer data into an integer array.
    """
    # Transformation unique identifier.
    _IDENTIFIER = "unravel.dataset.test_ints"

    def input(self: SelfTransdatasetUnravelInts, raw: Any, /) -> Input:
        r"""
        Convert raw data into input to the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Nothing to be unraveled.
            return [
                DatasetInts.from_memalias(
                    [DataInts.from_numeric({"content": np.array([], dtype=np.int64)})], ["full"]
                )
            ]
        else:
            # All the other cases are not supported.
            raise lib.transforms.ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into input domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def output(self: SelfTransdatasetUnravelInts, raw: Any, /) -> Output:
        r"""
        Convert raw data into output from the transformation.

        Args
        ----
        - raw
            Raw data.

        Returns
        -------
        - process
            Processed data compatible with the transformation.
        """
        # Conversion will vary according to raw data.
        if raw is None:
            # Nothing to be unraveled.
            return [np.array([], dtype=np.int64)]
        else:
            # All the other cases are not supported.
            raise lib.transforms.ErrorTransformUnsupportPartial(
                f"Try to formalize incompatible raw data into output domain of"
                f' "{self._IDENTIFIER:s}".'
            )

    def transform(
        self: SelfTransdatasetUnravelInts, input: Input, /, *args: Any, **kwargs: Any
    ) -> Output:
        r"""
        Transform input into output without inplacement.

        Args
        ----
        - input
            Input to the transformation.

        Returns
        -------
        - output
            Output from the transformation.
        """
        # Flatten all data container content into a collection of arraies.
        (dataset,) = input
        memory: List[NPINTS]
        memory = [np.array(data._content, dtype=np.int64) for data in dataset.memory]
        return memory

    def get_alphabetic_data(self: SelfTransdatasetUnravelInts, /) -> Mapping[str, Any]:
        r"""
        Get alphabetic data of the transformation.

        Args
        ----

        Returns
        -------
        - data
            Alphabetic data of the transformation.
        """
        # Do nothing.
        return {}

    def set_alphabetic_data(
        self: SelfTransdatasetUnravelInts, data: Mapping[str, Any], /  # noqa: W504
    ) -> SelfTransdatasetUnravelInts:
        r"""
        Set alphabetic data of the transformation.

        Args
        ----
        - data
            Alphabetic data of the transformation.

        Returns
        -------
        - self
            Class instance itself.
        """
        # Do nothing.
        return self
