# Import Python packages.
import os
from typing import Any

# Import PyTest packagtes.
import pytest

# Import PyTest external packages.
from py._path.local import LocalPath

# Import developing library.
import fin_tech_py_toolkit as lib
from fin_tech_py_toolkit.io import WSP

# Import testing library.
from .supp_base import (
    TransformIdentityComposition,
    TransformIdentityCompositionRedundant,
    TransformNull,
)


def test_null() -> None:
    r"""
    Test transformation with no functionality.

    Args
    ----

    Returns
    -------
    """
    # Test transformation tags.
    transform = TransformNull()
    tags = transform.tags
    assert not tags.inplacable
    assert not tags.invertible
    assert not tags.parametric
    assert not tags.hierarchy


def test_identity_composition(*, tmpdir: LocalPath) -> None:
    r"""
    Test transformation for identity composition.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Test transformation tags.
    root = str(tmpdir)
    transform = TransformIdentityComposition(
        children={
            "identity1": lib.transforms.TransformIdentity(),
            "identity2": lib.transforms.TransformIdentity(),
        }
    )
    tags = transform.tags
    assert tags.inplacable
    assert tags.invertible
    assert tags.parametric
    assert tags.hierarchy
    assert all(not child.tags.hierarchy for child in transform._children.values())

    # Test regular usage functions.
    input = transform.input(None)
    output = transform.output(None)
    transform.transform(input)
    transform.transform_(input)
    transform.inverse(output)
    transform.inverse_(output)
    transform.fit(input, output)
    transform.fit_transform((input, output), input)
    transform.fit_transform_((input, output), input)
    transform.fit_inverse((input, output), output)
    transform.fit_inverse_((input, output), output)
    transform.save(os.path.join(root, "identity_composition"))
    transform.save(os.path.join(root, "identity_composition"))
    transform.load(os.path.join(root, "identity_composition"))

    # Scan for generated backup.
    (path,) = [path for path in os.listdir(root) if path.startswith("identity_composition" + WSP)]
    origin = "identity_composition"
    backup = os.path.basename(path)

    # Test saving status.
    for title in [origin, backup]:
        # Check origin and backup in the same way.
        for path in [
            os.path.join(root, title),
            os.path.join(root, title, "identity1"),
            os.path.join(root, title, "identity2"),
        ]:
            # Check existence of saving data.
            assert os.path.isdir(path)
            assert os.path.isfile(os.path.join(path, "metadata.json"))
            assert os.path.isfile(os.path.join(path, "numeric.npz"))
            assert os.path.isfile(os.path.join(path, "alphabetic.json"))
            assert os.path.isfile(os.path.join(path, "children.json"))


@pytest.mark.parametrize("copy", [pytest.param(True, id="copy"), pytest.param(False, id="rename")])
def test_identity_composition_backup(*, tmpdir: LocalPath, copy: bool) -> None:
    r"""
    Test transformation for identity composition with backup saving.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.
    - copy
        If True, make a copy to generate backup.
        Otherwise, rename to generate backup.

    Returns
    -------
    """
    # Save data to file system.
    root = str(tmpdir)
    transform = TransformIdentityComposition(
        children={
            "identity1": lib.transforms.TransformIdentity(),
            "identity2": lib.transforms.TransformIdentity(),
        }
    )
    transform.save(os.path.join(root, "identity_composition"))

    # Make a duplicate save to trigger backup according to arguments.
    transform.backup(os.path.join(root, "identity_composition"), copy=copy)

    # Scan for generated backup.
    (path,) = [path for path in os.listdir(root) if path.startswith("identity_composition" + WSP)]
    origin = "identity_composition"
    backup = os.path.basename(path)

    # Origin saving status differs according to backup mode.
    for path in [
        os.path.join(root, origin),
        os.path.join(root, origin, "identity1"),
        os.path.join(root, origin, "identity2"),
    ]:
        # Check existence of saving data.
        assert copy == os.path.isdir(path)
        assert copy == os.path.isfile(os.path.join(path, "metadata.json"))
        assert copy == os.path.isfile(os.path.join(path, "numeric.npz"))
        assert copy == os.path.isfile(os.path.join(path, "alphabetic.json"))
        assert copy == os.path.isfile(os.path.join(path, "children.json"))

    # Test backup saving status.
    for path in [
        os.path.join(root, backup),
        os.path.join(root, backup, "identity1"),
        os.path.join(root, backup, "identity2"),
    ]:
        # Check existence of saving data.
        assert os.path.isdir(path)
        assert os.path.isfile(os.path.join(path, "metadata.json"))
        assert os.path.isfile(os.path.join(path, "numeric.npz"))
        assert os.path.isfile(os.path.join(path, "alphabetic.json"))
        assert os.path.isfile(os.path.join(path, "children.json"))


@pytest.mark.xfail(raises=lib.transforms.ErrorTransformInvalidDefinition)
def test_identity_composition_get_child_redundant() -> None:
    r"""
    Test transformation for identity composition with redundant on getting child method.

    Args
    ----

    Returns
    -------
    """
    # Create transformation with improper children.
    TransformIdentityCompositionRedundant(
        children={
            "identity1": lib.transforms.TransformIdentity(),
            "identity2": lib.transforms.TransformIdentity(),
        }
    )


@pytest.mark.parametrize(
    ("identity1", "identity2"),
    [
        pytest.param(
            lib.transforms.TransformIdentity(),
            None,
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformInvalidDefinition)],
            id="not-transform",
        ),
        pytest.param(
            lib.transforms.TransformIdentity(),
            TransformNull(),
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformInvalidDefinition)],
            id="diff-annotate",
        ),
    ],
)
def test_identity_composition_get_child_type_error(*, identity1: Any, identity2: Any) -> None:
    r"""
    Test transformation for identity composition with type error on getting child method.

    Args
    ----

    Returns
    -------
    """
    # Create transformation with improper children.
    TransformIdentityComposition(children={"identity1": identity1, "identity2": identity2})


@pytest.mark.parametrize(
    "explicit", [pytest.param(True, id="explicit"), pytest.param(False, id="temporary")]
)
def test_null_cache(*, tmpdir: LocalPath, explicit: bool) -> None:
    r"""
    Test transformation with no functionality on explicit caching.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.
    - explicit
        If True, the cache will be an explicit cache.

    Returns
    -------
    """
    # Specify caching directory.
    root = str(tmpdir)
    root_cache = os.path.join(root, "cache")
    name_cache = "explicit" if explicit else "temporary"
    cache = os.path.join(root_cache, TransformNull._IDENTIFIER, name_cache)

    # Create cache automatically on creation.
    transform = TransformNull(
        cache_prefix=root_cache if explicit else None, cache_suffix=name_cache
    )
    assert os.path.isdir(cache) == explicit

    # Explicit cache should not be automatically on deletion.
    del transform
    assert os.path.isdir(cache) == explicit
