# Import Python packages.
import copy
import itertools
from typing import Any, List, Mapping, Tuple

# Import external packages.
import pandas as pd

# Import PyTest packagtes.
import pytest

# Import PyTest external packages.
from py._path.local import LocalPath

# Import developing library.
import fin_tech_py_toolkit as lib

# Import testing library.
from ...utils import eq_dataframe, to_eq_plural_ordered
from ..utils import template_test_io, template_test_transform


# Type aliases.
Input = List[pd.DataFrame]
Output = List[pd.DataFrame]


# Runtime constants.
IDENTIFIER = lib.transforms.TransformTabularize._IDENTIFIER


def synthesize(*, enforce: bool) -> Tuple[Tuple[Input, Output], Input, Output, Mapping[str, Any]]:
    r"""
    Synthesize test I/O.

    Args
    ----
    - enforce
        If True, output case will correspond to enforcing arbitrary columns to be categorical.

    Returns
    -------
    - example
        Input and output examples.
    - input
        Input case.
    - output
        Output case.
    - supplement
        Supplementary materical for synthesized test.
    """
    # Create data series of different common value types.
    strings = ['"X"', '"X"', '"Y"', '"Y"', '"Y"']
    nans = [float("nan"), float("nan"), float("nan"), float("nan"), float("nan")]
    ints = [1, 2, 3, 4, 5]
    floats = [float("nan"), float("-inf"), 0.0, 1.0, float("inf")]
    ids = [10011, 10011, float("nan"), 10022, 10022]

    # Define default column value tyes.
    categorical = ["string", "nan"]
    continuous = ["int", "float"]
    discretizable = ["id"]

    # Decide final column value types based on flags.
    categorical = categorical + discretizable if enforce else categorical
    continuous = continuous if enforce else continuous + discretizable
    discretizable = discretizable if enforce else []

    # Create a dataframe including common value types.
    dataframe = pd.DataFrame(
        {"string": strings, "nan": nans, "int": ints, "float": floats, "id": ids}
    )

    # Create input and output examples.
    example_input: Input
    example_input = [dataframe]
    example_output: Output
    example_output = []

    # Input case.
    input: Input
    input = [dataframe]

    # Output case.
    output: Output
    output = [dataframe[categorical], dataframe[continuous]]
    return (example_input, example_output), input, output, {"discretizable": discretizable}


@pytest.mark.parametrize(
    ("raw_input", "raw_output"),
    [
        pytest.param(
            ...,
            None,
            id="unsupport-input",
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        ),
        pytest.param(
            None,
            ...,
            id="unsupport-output",
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        ),
        pytest.param(None, None, id="both-null"),
    ],
)
def test_io(*, raw_input: Any, raw_output: Any) -> None:
    r"""
    Test transformation input and output domain formalization.

    Args
    ----
    - raw_input
        Raw input.
    - raw_output
        Raw output.

    Returns
    -------
    """
    # Initialize testing transformation.
    factory = lib.transforms.FactoryTransform()

    # Run test template.
    template_test_io(
        IDENTIFIER,
        factory,
        raw_input,
        raw_output,
        to_eq_plural_ordered(eq_dataframe),
        to_eq_plural_ordered(eq_dataframe),
    )


@pytest.mark.parametrize(
    "enforce", [pytest.param(True, id="enforce-category"), pytest.param(False, id="automate")]
)
def test_default(*, tmpdir: LocalPath, enforce: bool) -> None:
    r"""
    Test transformation for tabularization.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.
    - enforce
        If True, output case will correspond to enforcing arbitrary columns to be categorical.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.transforms.FactoryTransform()

    # Generate inputs and outputs.
    example, input, output, supplement = synthesize(enforce=enforce)

    # Run test template.
    template_test_transform(
        root,
        IDENTIFIER,
        factory,
        example,
        input,
        output,
        to_eq_plural_ordered(eq_dataframe),
        to_eq_plural_ordered(eq_dataframe),
        fit_kwargs=dict(discretizable=supplement["discretizable"]),
    )


def test_empty(*, tmpdir: LocalPath) -> None:
    r"""
    Test transformation for tabularization on empty Pandas data.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.transforms.FactoryTransform()

    # Generate inputs and outputs.
    transform = factory.from_args(IDENTIFIER)
    input = transform.input(None)
    output = transform.output(None)
    example: Tuple[Input, Output]
    example = (input, [])

    # Run test template.
    template_test_transform(
        root,
        IDENTIFIER,
        factory,
        example,
        input,
        output,
        to_eq_plural_ordered(eq_dataframe),
        to_eq_plural_ordered(eq_dataframe),
    )


@pytest.mark.parametrize(
    ("redundant", "inverse"),
    [
        pytest.param(
            redundant,
            inverse,
            id="{:s}-{:s}".format(redundant, "inverse" if inverse else "transform"),
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        )
        for redundant, inverse in itertools.product(["categorical", "continuous"], [True, False])
    ],
)
def test_redundant_columns(*, tmpdir: LocalPath, redundant: str, inverse: bool) -> None:
    r"""
    Test transformation for tabularization with redundant columns.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.
    - redundant
        Redundant column value type.
    - inverse
        If True, test only inversion, otherwise, test only transformation.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.transforms.FactoryTransform()

    # Generate inputs and outputs.
    generic = pd.DataFrame({"categorical": ["A"], "continuous": [0]})
    generic[f"{redundant:s}_"] = generic[redundant]
    categorical = ["categorical"] + (["categorical_"] if redundant == "categorical" else [])
    continuous = ["continuous"] + (["continuous_"] if redundant == "continuous" else [])
    input = [generic]
    output = [generic[categorical], generic[continuous]]
    input_ = copy.deepcopy(input)
    del input_[0][f"{redundant:s}_"]
    example: Tuple[Input, Output]
    example = (input_, [])

    # Run test template.
    template_test_transform(
        root,
        IDENTIFIER,
        factory,
        example,
        input,
        output,
        to_eq_plural_ordered(eq_dataframe),
        to_eq_plural_ordered(eq_dataframe),
        require_test_transform=not inverse,
        require_test_transform_=not inverse,
        require_test_inverse=inverse,
        require_test_inverse_=inverse,
    )


@pytest.mark.parametrize(
    ("missing", "inverse"),
    [
        pytest.param(
            missing,
            inverse,
            id="{:s}-{:s}".format(missing, "inverse" if inverse else "transform"),
            marks=[pytest.mark.xfail(raises=lib.transforms.ErrorTransformUnsupportPartial)],
        )
        for missing, inverse in itertools.product(["categorical", "continuous"], [True, False])
    ],
)
def test_missing_columns(*, tmpdir: LocalPath, missing: str, inverse: bool) -> None:
    r"""
    Test transformation for tabularization with missing columns.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.
    - missing
        Missing column value type.
    - inverse
        If True, test only inversion, otherwise, test only transformation.

    Returns
    -------
    """
    # Initialize testing transformation.
    root = str(tmpdir)
    factory = lib.transforms.FactoryTransform()

    # Generate inputs and outputs.
    generic = pd.DataFrame({"categorical": ["A"], "continuous": [0]})
    categorical = [] if missing == "categorical" else ["categorical"]
    continuous = [] if missing == "continuous" else ["continuous"]
    input_ = [generic]
    input = copy.deepcopy(input_)
    del input[0][missing]
    output = [generic[categorical], generic[continuous]]
    example: Tuple[Input, Output]
    example = (input_, [])

    # Run test template.
    template_test_transform(
        root,
        IDENTIFIER,
        factory,
        example,
        input,
        output,
        to_eq_plural_ordered(eq_dataframe),
        to_eq_plural_ordered(eq_dataframe),
        require_test_transform=not inverse,
        require_test_transform_=not inverse,
        require_test_inverse=inverse,
        require_test_inverse_=inverse,
    )
