# Import Python packages.
import filecmp
import os

# Import external packages.
import more_itertools as xitertools

# Import PyTest external packages.
from py._path.local import LocalPath

# Import developing library.
import fin_tech_py_toolkit as lib

# Import testing library.
from .base.supp_base import TransformIdentityComposition


def test_factory_reproduce_identity(*, tmpdir: LocalPath) -> None:
    r"""
    Test transformation factory reproducing identity transformation.

    Args
    ----
    - tmpdir
        Temporary directory for this test.
        It is automatically provided by PyTest, so its value should not be explicitly defined.

    Returns
    -------
    """
    # Save data to file system.
    root = str(tmpdir)
    transform1 = TransformIdentityComposition(
        children={
            "identity1": lib.transforms.TransformIdentity(),
            "identity2": lib.transforms.TransformIdentity(),
        }
    )
    transform1.save(os.path.join(root, "__init__"))

    # Use factory to generate another transformation.
    factory = lib.transforms.FactoryTransform()
    transform2 = factory.from_filesys(os.path.join(root, "__init__"))
    transform2.save(os.path.join(root, "from_filesys"))

    # Collect data on file system.
    directory = os.path.join(root, "__init__")
    data1 = list(
        sorted(
            xitertools.flatten(
                [
                    [
                        os.path.relpath(os.path.join(prefix, suffix), directory)
                        for suffix in suffices
                    ]
                    for prefix, _, suffices in os.walk(directory)
                ]
            )
        )
    )
    directory = os.path.join(root, "from_filesys")
    data2 = list(
        sorted(
            xitertools.flatten(
                [
                    [
                        os.path.relpath(os.path.join(prefix, suffix), directory)
                        for suffix in suffices
                    ]
                    for prefix, _, suffices in os.walk(directory)
                ]
            )
        )
    )

    # Compare two transformations.
    assert type(transform1) is type(transform2)
    assert len(data1) == len(data2)
    for path1, path2 in zip(data1, data2):
        # Relative path and binary content must match between every saved data file of two
        # transformations.
        assert path1 == path2
        assert filecmp.cmp(
            os.path.join(root, "__init__", path1), os.path.join(root, "from_filesys", path2)
        )
