// based on the code from https://en.wikipedia.org/wiki/2-opt?ysclid=mfyyxs69w4580185422

#include <random>
#include <stdio.h>
#include <vector>
#include <iostream>
#include <string>
#include <cstdlib>
#include <fstream>
#include <chrono>
#include <iomanip>
#include <fstream>

using namespace std;
using namespace std::chrono;

double mean(const std::vector<double>& numbers) {
    double sum = 0.0;
    for (double num : numbers) {
        sum += num;
    }
    return numbers.empty() ? 0 : sum / numbers.size();
}

double standard_dev(const std::vector<double>& numbers) {
    double mean_value = mean(numbers);
    double sum = 0.0;
    for (double num : numbers) {
        sum += pow(num - mean_value, 2);
    }
    return numbers.empty() ? 0 : sqrt(sum / (numbers.size() - 1));
}


class Point {
public:
  double x, y;

  Point(double x, double y) {
    this->x = x;
    this->y = y;
  }
  Point() {
    this->x = 0.0;
    this->y = 0.0;
  }

  // Distance between two points
  inline double dist(const Point &other) const {
    double diffx = x - other.x;
    double diffy = y - other.y;
    return sqrt(diffx * diffx + diffy * diffy);
  }
};

// Calculate the distance of the whole circuit
double pathLength(const vector<Point> &problem, const vector<int> &path) {
    int n = path.size();
    double length = problem[path[n - 1]].dist(problem[path[0]]);
    for (int i = 0; i < n - 1; i++) {
        length += problem[path[i]].dist(problem[path[i + 1]]);
    }
    return length;
}

// Replace edges path[i-1]->path[i] and path[j]->path[j+1]
// with path[i-1]->path[j] and path[i]->path[j+1]
void swap_edges(vector<int> &path, int i, int j) {
  while (i < j) {
    int temp = path[i];
    path[i] = path[j];
    path[j] = temp;
    i++;
    j--;
  }
}

int run_2_opt(const vector<Point> &problem, vector<int> &path, int max_iterations) {
    double curLength = pathLength(problem, path);
    int n = path.size();
    int iteration = 0;
    bool foundImprovement = true;
    while (foundImprovement && iteration < max_iterations) {
        foundImprovement = false;
        for (int i = 1; i < n; i++) {
            for (int j = i + 1; j < n; j++) {
                double lengthDelta =
                -problem[path[i - 1]].dist(problem[path[i]]) + problem[path[i - 1]].dist(problem[path[j]])
                -problem[path[j]].dist(problem[path[(j + 1) % n]]) + problem[path[i]].dist(problem[path[(j + 1) % n]]);

                // If the length of the path is reduced, do a 2-opt swap
                if (lengthDelta < 0) {
                    swap_edges(path, i, j);
                    curLength += lengthDelta;
                    foundImprovement = true;
                    ++iteration;
                    break;
                }
            }
            if (foundImprovement) {
                break;
            }
        }
    }
    if (iteration >= max_iterations) {
        cout << "MAX ITERATIONS EXCEEDED" << endl;
    }
    return iteration;
}

void load_init_tours(string filename, int problem_size, int num_problems, vector<vector<Point>> &problems, vector<vector<int>> &init_paths) {
    ifstream file (filename);
    double x, y;
    string sep;
    int index;
    if (file.is_open()) {
        for (int i = 0; i < num_problems; ++i) {
            for (int j = 0; j < problem_size; ++j) {
                file >> x >> y;
                problems[i][j] = Point(x, y);
            }
            file >> sep;
            for (int j = 0; j < problem_size; ++j) {
                file >> index;
                init_paths[i][j] = index;
            }
        }
        file.close();
    }
}

void print_configuration(string filename, int problem_size, int num_problems, int max_iterations) {
    cout << "CONFIGURATION HYPERPARAMETERS" << endl;
    cout << "Filename: " << filename << endl;
    cout << "Problem size: " << problem_size << endl;
    cout << "Number of problems: " << num_problems << endl;
    cout << "Max iterations: " << max_iterations << endl;
    cout << endl;
}

int main(int argc, char* argv[]) {
    // load greedy tours for all problems
    string tour_filename = argv[1];
    int problem_size = atoi(argv[2]);
    int num_problems = atoi(argv[3]);
    int max_iterations = atoi(argv[4]);
    string save_filename_prefix = argv[5];

    string opt_tour_filename = save_filename_prefix + "_" + to_string(max_iterations) + "_tour.txt";
    string dist_filename = save_filename_prefix + "_" + to_string(max_iterations) + "_dist.txt";
    print_configuration(tour_filename, problem_size, num_problems, max_iterations);
    
    vector<vector<Point>> problems(num_problems, vector<Point>(problem_size));
    vector<vector<int>> paths(num_problems, vector<int>(problem_size));
    load_init_tours(tour_filename, problem_size, num_problems, problems, paths);

    vector<double> init_len_vector(num_problems);
    vector<double> opt_len_vector(num_problems);
    vector<double> duration_vector(num_problems);
    vector<double> iter_vector(num_problems);
    
    auto start_time_total = high_resolution_clock::now();
    for (int i = 0; i < num_problems; ++i) {
        cout << "RUN " << i << endl;
        double init_len = pathLength(problems[i], paths[i]);
        init_len_vector[i] = init_len;
        
        auto run_start_time = high_resolution_clock::now();
        int run_iter = run_2_opt(problems[i], paths[i], max_iterations);
        auto run_stop_time = high_resolution_clock::now();
        auto run_duration = duration_cast<milliseconds>(run_stop_time - run_start_time);
        duration_vector[i] = run_duration.count();
        iter_vector[i] = run_iter;
        
        cout << "Time elapsed (milliseconds): " << run_duration.count() << endl;
        cout << "Num iterations: " << run_iter << endl;
        double opt_len = pathLength(problems[i], paths[i]);
        opt_len_vector[i] = opt_len;
        cout << init_len << " " << opt_len << endl;
        cout << endl;
    }
    auto stop_time_total = high_resolution_clock::now();
    auto duration_total = duration_cast<seconds>(stop_time_total - start_time_total);

    cout << "Avg. Init Tour Length: " << mean(init_len_vector) << " ± " << standard_dev(init_len_vector) << endl;
    cout << "Avg. Optimized Tour Length: " << mean(opt_len_vector) << " ± " << standard_dev(opt_len_vector) << endl;
    cout << "Avg. Time (milliseconds): " << mean(duration_vector) << " ± " << standard_dev(duration_vector) << endl;
    cout << "Total Time (seconds): " << duration_total.count() << endl;
    cout << "Avg. Num Iterations: " << mean(iter_vector) << " ± " << standard_dev(iter_vector) << endl;

    ofstream out_file;
    out_file.open(opt_tour_filename);
    if (out_file.is_open()) {
        for (int i = 0; i < num_problems; i++) {
            for (int j = 0; j < problem_size; j++) {
                out_file << paths[i][j] << " ";
            }
            out_file << "\n";
        }
    }
    out_file.close();
    
    return 0;
}
