// based on the code from https://github.com/ArGintum/RTD-Lite

#include <iostream>
#include <vector>
#include <climits>
#include <queue>
#include <map>
#include <string>
#include <algorithm>
#include <numeric>
#include <fstream>
#include <iomanip>

using namespace std;

class DSU {
  public:
	vector<int> parent;
	vector<int> rank;

	DSU(int n_vertices) : parent(n_vertices), rank(n_vertices, 0) {
        iota(parent.begin(), parent.end(), 0);
	}

    DSU(const DSU &copy_from) {
        parent.assign(copy_from.parent.begin(), copy_from.parent.end());
        rank.assign(copy_from.rank.begin(), copy_from.rank.end());
    }

	int find(int v) {
        if (parent[v] == v) {
            return v;
        }
        return parent[v] = find(parent[v]);
    }

    void unite(int u, int v) {
        int u_root = find(u);
        int v_root = find(v);
        if (u_root != v_root) {
            if (rank[u_root] < rank[v_root]) {
                swap(u_root, v_root);
            }
            if (rank[u_root] == rank[v_root]) {
                ++rank[u_root];
            }
            parent[v_root] = u_root;
        }
    }
};

void prim_algo(const vector<vector<double>> &adjacency_matrix, vector<vector<int>> &mst_edges, vector<double> &edge_weights) {
    int n = adjacency_matrix.size();

    vector<int> ancestors(n, -1);
    vector<double> dst(n, INT_MAX);
    vector<bool> visited(n, false);

    dst[0] = 0;
    ancestors[0] = -1;
    int v = 0;

    for (int iter = 0; iter < n; ++iter) {
        visited[v] = true;
        double mn_dst = INT_MAX;
        int next_vertex = -1;

        for (int i = 0; i < n; i++) {
            if (!visited[i] && adjacency_matrix[v][i] < dst[i]) {
                dst[i] = adjacency_matrix[v][i];
                ancestors[i] = v;
            }
            if (!visited[i] && dst[i] < mn_dst) {
                next_vertex = i;
                mn_dst = dst[i];
            }
        }

        // Failsafe for multiple connected components
        if (next_vertex == -1) {
            break;
        }
        v = next_vertex;
    }

    for (int i = 1; i < n; i++) {
        mst_edges[i - 1][0] = i;
        mst_edges[i - 1][1] = ancestors[i];
        edge_weights[i - 1] = adjacency_matrix[i][ancestors[i]];
    }
}

class RTD_Lite_TSP_Penalty {
    private:
        int n;
        vector<vector<double>> rmin;
        vector<vector<int>> rmin_edge_idx;
        vector<double> rmin_edge_w;

        vector<vector<int>> tour_edge_idx;
        vector<double> tour_edge_w;
        vector<int> sort_idx;
        vector<vector<int>> r1_edge_idx;
        vector<double> r1_edge_w;

        void sort_edges_weights(vector<vector<int>> &edge_idx, vector<double> &edge_w) {
            vector<int> idx(n - 1);
            iota(idx.begin(), idx.end(), 0);
            sort(idx.begin(), idx.end(), [&edge_w](int i, int j) {return edge_w[i] < edge_w[j];});

            vector<vector<int>> tmp_edge_idx;
            tmp_edge_idx.assign(edge_idx.begin(), edge_idx.end());

            for (int i = 0; i < n - 1; i++) {
                edge_idx[i] = tmp_edge_idx[idx[i]];
            }

            vector<double> tmp_edge_w;
            tmp_edge_w.assign(edge_w.begin(), edge_w.end());
            for (int i = 0; i < n - 1; i++) {
                edge_w[i] = tmp_edge_w[idx[i]];
            }
        }

        void create_tour_edge_idx(const vector<int> &tour) {
            for (int i = 0; i < n; i++) {
                tour_edge_idx[i][0] = tour[i];
                tour_edge_idx[i][1] = tour[(i + 1) % n];
            }
        }

        void create_tour_edge_weights(const vector<int> &tour) {
            for (int i = 0; i < n; i++) {
                tour_edge_w[i] = rmin[tour[i]][tour[(i + 1) % n]];
            }
        }

        void prim_algo_tour() {
            iota(sort_idx.begin(), sort_idx.end(), 0);
            sort(sort_idx.begin(), sort_idx.end(), [this](int i, int j) {return tour_edge_w[i] < tour_edge_w[j];});
            for (int i = 0; i < n - 1; i++) {
                r1_edge_idx[i] = tour_edge_idx[sort_idx[i]];
                r1_edge_w[i] = tour_edge_w[sort_idx[i]];
            }
        }

    public:

        RTD_Lite_TSP_Penalty(const vector<vector<double>> &rmin) : rmin(rmin){
            n = rmin.size();
            rmin_edge_idx.resize(n - 1, vector<int>(2));
            rmin_edge_w.resize(n - 1);
            prim_algo(rmin, rmin_edge_idx, rmin_edge_w);
            sort_edges_weights(rmin_edge_idx, rmin_edge_w);

            tour_edge_idx.resize(n, vector<int>(2));
            tour_edge_w.resize(n);
            sort_idx.resize(n);
            r1_edge_idx.resize(n - 1, vector<int>(2));
            r1_edge_w.resize(n - 1);
        }

        void run(const vector<int> &tour, vector<pair<vector<int>, double>> &edge_penalty) {
            create_tour_edge_idx(tour);
            create_tour_edge_weights(tour);
            prim_algo_tour();
            double min_penalty = INT_MAX;

            DSU min_graph_dsu(n);
            for (int i = 0; i < n - 1; i++) {
                int u_clique = min_graph_dsu.find(rmin_edge_idx[i][0]);
                int v_clique = min_graph_dsu.find(rmin_edge_idx[i][1]);
                double birth = rmin_edge_w[i];
                vector<int> birth_idx = rmin_edge_idx[i];

                DSU r1_graph_dsu(min_graph_dsu);
                double death_1;
                vector<int> death_1_idx;
                for (int j = 0; j < n - 1; j++) {
                    r1_graph_dsu.unite(r1_edge_idx[j][0], r1_edge_idx[j][1]);
                    if (r1_graph_dsu.find(u_clique) == r1_graph_dsu.find(v_clique)) {
                        death_1 = r1_edge_w[j];
                        death_1_idx = r1_edge_idx[j];
                        break;
                    }
                }

                if (death_1 > birth && death_1 != INT_MAX) {
                    edge_penalty.push_back({death_1_idx, death_1 - birth});
                    min_penalty = min(min_penalty, death_1 - birth);
                }

                min_graph_dsu.unite(rmin_edge_idx[i][0], rmin_edge_idx[i][1]);
            }
        }
};
