"""Simple travelling salesman problem on a circuit board."""

from __future__ import print_function

import math

from ortools.constraint_solver import pywrapcp
from ortools.constraint_solver import routing_enums_pb2


def get_index_cost(plan_output):
    cost = 0
    for i in range(len(plan_output) - 1):
        cost += math.pow(plan_output[i + 1] - plan_output[i], 2)
    return cost


def create_data_model(instances):
    """Stores the data for the problem."""
    data = {"locations": instances, "num_vehicles": 1, "depot": 0}
    # Locations in block units
    return data


def compute_euclidean_distance_matrix(locations):
    """Creates callback to return distance between points."""
    distances = {}
    for from_counter, from_node in enumerate(locations):
        distances[from_counter] = {}
        for to_counter, to_node in enumerate(locations):
            if from_counter == to_counter:
                distances[from_counter][to_counter] = 0
            else:
                # Euclidean distance
                distances[from_counter][to_counter] = int(
                    math.hypot((from_node[0] - to_node[0]), (from_node[1] - to_node[1]))
                )
    return distances


def print_solution(manager, routing, solution):
    """Prints solution on console."""
    print("Objective: {}".format(solution.ObjectiveValue()))
    index = routing.Start(0)
    plan_output = "Route:\n"
    route_distance = 0
    while not routing.IsEnd(index):
        plan_output += " {} ->".format(manager.IndexToNode(index))
        previous_index = index
        index = solution.Value(routing.NextVar(index))
        route_distance += routing.GetArcCostForVehicle(previous_index, index, 0)
    plan_output += " {}\n".format(manager.IndexToNode(index))
    print(plan_output)
    plan_output += "Objective: {}m\n".format(route_distance)


def solve(instance, return_path=False) -> float | tuple[float, list[int]]:
    """Entry point of the program."""
    # Instantiate the data problem.
    data = create_data_model(instance)

    # Create the routing index manager.
    manager = pywrapcp.RoutingIndexManager(len(data["locations"]), data["num_vehicles"], data["depot"])

    # Create Routing Model.
    routing = pywrapcp.RoutingModel(manager)

    distance_matrix = compute_euclidean_distance_matrix(data["locations"])

    def distance_callback(from_index, to_index):
        """Returns the distance between the two nodes."""
        # Convert from routing variable Index to distance matrix NodeIndex.
        from_node = manager.IndexToNode(from_index)
        to_node = manager.IndexToNode(to_index)
        return distance_matrix[from_node][to_node]

    transit_callback_index = routing.RegisterTransitCallback(distance_callback)

    # Define cost of each arc.
    routing.SetArcCostEvaluatorOfAllVehicles(transit_callback_index)

    # Setting first solution heuristic.
    search_parameters = pywrapcp.DefaultRoutingSearchParameters()
    search_parameters.first_solution_strategy = routing_enums_pb2.FirstSolutionStrategy.PATH_CHEAPEST_ARC

    # Solve the problem.
    solution = routing.SolveWithParameters(search_parameters)

    # Get route
    plan_output = []
    index = routing.Start(0)
    while not routing.IsEnd(index):
        plan_output.append(manager.IndexToNode(index))
        index = solution.Value(routing.NextVar(index))
    plan_output.append(manager.IndexToNode(index))

    # Print solution on console.
    # if solution:
    #     print_solution(manager, routing, solution)

    if not return_path:
        return solution.ObjectiveValue()
    else:
        return solution.ObjectiveValue(), plan_output


def my_solve(instance, city_indices):
    """Entry point of the program."""
    # Instantiate the data problem.
    data = create_data_model(instance)
    # Create the routing index manager.
    manager = pywrapcp.RoutingIndexManager(len(data["locations"]), data["num_vehicles"], data["depot"])

    # Create Routing Model.
    routing = pywrapcp.RoutingModel(manager)

    distance_matrix = compute_euclidean_distance_matrix(data["locations"])

    def distance_callback(from_index, to_index):
        """Returns the distance between the two nodes."""
        # Convert from routing variable Index to distance matrix NodeIndex.
        from_node = manager.IndexToNode(from_index)
        to_node = manager.IndexToNode(to_index)
        return distance_matrix[from_node][to_node]

    transit_callback_index = routing.RegisterTransitCallback(distance_callback)

    # Define cost of each arc.
    routing.SetArcCostEvaluatorOfAllVehicles(transit_callback_index)

    # Setting first solution heuristic.
    search_parameters = pywrapcp.DefaultRoutingSearchParameters()
    search_parameters.first_solution_strategy = routing_enums_pb2.FirstSolutionStrategy.PATH_CHEAPEST_ARC

    # Solve the problem.
    solution = routing.SolveWithParameters(search_parameters)

    # Get route
    plan_output = []
    index = routing.Start(0)
    while not routing.IsEnd(index):
        plan_output.append(city_indices[manager.IndexToNode(index)])
        index = solution.Value(routing.NextVar(index))
    plan_output.append(city_indices[manager.IndexToNode(index)])
    # Print solution on console.
    # print_solution(manager, routing, solution)

    return plan_output, solution.ObjectiveValue() / 1000
