import argparse
import glob
import os

from cli.lkh_postprocess import main as lkh_postprocess
from tqdm import tqdm


def main(args: argparse.Namespace) -> None:
    target_postfix = ".tsp"
    target_files = [
        file
        for file in glob.glob(f"{args.input_folder}/*")
        if target_postfix == file[-4:]
        and f"{args.n_node}node" in file
        and args.location in file
        and args.keyword in file
    ]

    if len(target_files) == 0:
        print(args.input_folder)
        raise ValueError("input files is missing")

    box = []
    ls_box = []
    duration_box = []
    for file in tqdm(target_files):
        args_dict = {
            "problem_input": file,
            "solution_input": file.replace(target_postfix, ".txt"),
            "output": args.output,
        }
        tmp_args = argparse.Namespace(**args_dict)
        cost_box, ls_cost_box, duration = lkh_postprocess(tmp_args)
        box.append(max(cost_box))
        ls_box.append(max(ls_cost_box))
        duration_box.append(duration)

    assert len(box) == args.num_instance
    assert len(ls_box) == args.num_instance

    basename = "_".join(os.path.basename(file).split("_")[:-1])
    filename = f"{args.output}/{basename}.txt"
    with open(filename, "w") as f:
        for line in box:
            print(line, file=f)
        print("ave", round(sum(box) / len(box), 3), file=f)
    print(filename)

    filename = f"{args.output}/LS_{basename}.txt"
    with open(filename, "w") as f:
        for line in ls_box:
            print(line, file=f)
        print("ave", round(sum(ls_box) / len(ls_box), 3), round(sum(duration_box) / len(duration_box), 3), file=f)
    print(filename)


if __name__ == "__main__":
    # torch.use_deterministic_algorithms(True)

    parser = argparse.ArgumentParser()
    parser.add_argument("-i", "--input-folder", type=str, required=True, help="path to problem file")
    parser.add_argument("--output", type=str, default="LKH_output", help="output-folder")
    parser.add_argument("--n-node", type=int, help="output-folder")
    parser.add_argument("--num-instance", type=int, help="output-folder")
    parser.add_argument("--location", type=str, help="location")
    parser.add_argument("--key_word", type=str, help="location")

    args = parser.parse_args()
    main(args)
