import pulp


def solve_tsp(dist, target_list, depot):
    target_list.append(depot)
    target_list = list(set(target_list))

    x = {(i, j): pulp.LpVariable(cat="Binary", name=f"x_{i}_{j}") for i in target_list for j in target_list if i != j}
    u = {i: pulp.LpVariable(cat="Integer", name=f"u_{i}", lowBound=1, upBound=len(target_list)) for i in target_list}

    problem = pulp.LpProblem(name="TSP", sense=pulp.LpMinimize)
    problem += pulp.lpSum([x[(i, j)] * dist[i][j] for i in target_list for j in target_list if i != j])

    for i in target_list:
        problem.addConstraint(pulp.lpSum([x[(i, j)] for j in target_list if j != i]) == 1, f"outflow_{i}")
        problem.addConstraint(pulp.lpSum([x[(j, i)] for j in target_list if j != i]) == 1, f"inflow_{i}")

    for i in target_list:
        for j in target_list:
            if i != depot and j != depot and i != j:
                problem.addConstraint(
                    u[i] - u[j] + (len(target_list) - 1) * x[i, j] <= len(target_list) - 2, f"up_{i}_{j}"
                )

    solver = pulp.GUROBI_CMD(msg=True, threads=1, timeLimit=60)
    status = problem.solve(solver)

    if pulp.LpStatus[status] != "Optimal":
        print("Infeasible!")
        breakpoint()
        exit()

    z_result = {i: pulp.value(u[i]) for i in target_list if i != depot}
    path = [v for v, _ in sorted(z_result.items(), key=lambda x: x[1])]
    path.insert(0, depot)
    path.append(depot)

    return path, round(problem.objective.value(), 3)
