import json


class PiiResult:
    def __init__(self, entity, start, end, orig, new_value):
        self.entity = entity
        self.orig = orig
        self.start = start
        self.end = end
        self.new_value = new_value

    @staticmethod
    def fromJSON(json_dict):
        return PiiResult(**json_dict)


class PiiResults:
    def __init__(self, results):
        self.results = results

    def get_unique_pii(self):
        return set([result.entity for result in self.results])

    def number_of_unique_pii(self):
        return len(self.get_unique_pii())

    def toJSON(self):
        return json.dumps(self, default=lambda o: o.__dict__, sort_keys=True, indent=4)

    def pii_score(self):
        unique_pii = self.number_of_unique_pii()
        total_pii = len(self.results)
        return unique_pii

    def summary_string(self):
        return f"pii_score={self.pii_score()}, unique pii={self.number_of_unique_pii()}"

    @staticmethod
    def fromJSON(json_dict):
        res = PiiResults([])

        for r in json_dict["results"]:
            res.results.append(PiiResult.fromJSON(r))

        return res


class Message:
    def __init__(self, sender, text, pii_results: PiiResults = None):
        self.text = text
        self.sender = sender
        self.pii_results = pii_results

    def set_pii_result(self, pii_results):
        self.pii_results = pii_results

    def get_pii_score(self):
        if self.pii_results is None:
            return 0.0
        return self.pii_results.pii_score() / max(1.0, len(self.text) / 200)

    def contains_pii(self):
        return self.get_pii_score() > 0.0

    def get_unique_pii(self):
        if self.pii_results is None:
            return set()
        else:
            u_pii = self.pii_results.get_unique_pii()
            return u_pii

    def toJSON(self):
        return json.dumps(self, default=lambda o: o.__dict__, sort_keys=True, indent=4)

    def __str__(self) -> str:
        res_str = f"=== Message: Sender={self.sender}, pii_results={self.pii_results.summary_string() if self.pii_results is not None else None} ===  \n\t\t {self.text})"
        return res_str

    @staticmethod
    def fromJSON(json_dict):
        sender = json_dict["sender"]
        text = json_dict["text"]
        pii_results = None

        if json_dict["pii_results"] is not None:
            pii_results = PiiResults.fromJSON(json_dict["pii_results"])

        msg = Message(sender=sender, text=text, pii_results=pii_results)

        return msg


class DataConversation:
    def __init__(self, id, origin, language, messages):
        self.id = id
        self.origin = origin
        self.language = language
        self.messages = messages

    def contains_pii(self):
        return self.get_pii_score() > 0.0

    def get_pii_score(self):
        pii_sum = 0.0
        for message in self.messages:
            pii_sum += message.get_pii_score()
        return pii_sum / len(self.messages)

    def get_unique_pii(self):
        u_pii = set()
        for message in self.messages:
            u_pii = u_pii.union(message.get_unique_pii())
        return u_pii

    def size(self):
        msg_sizes = [len(message.text) for message in self.messages]
        return sum(msg_sizes)

    def __str__(self) -> str:
        res = f"Conversation(id={self.id}, origin={self.origin}, language={self.language}, #messages={len(self.messages)})"
        for msg in self.messages:
            res += "\n\n\t" + str(msg)
        return res

    def toJSON(self):
        return json.dumps(self, default=lambda o: o.__dict__, sort_keys=True, indent=4)

    @staticmethod
    def fromJSON(json_dict):
        id = json_dict["id"]
        origin = json_dict["origin"]
        language = json_dict["language"]
        messages = []

        for msg in json_dict["messages"]:
            messages.append(Message.fromJSON(msg))

        conv = DataConversation(
            id=id, origin=origin, language=language, messages=messages
        )

        return conv


class ConversationSet:
    def __init__(self, conversations):
        self.conversations = conversations

    def contains_pii(self):
        raise NotImplementedError
        return any([conversation.contains_pii() for conversation in self.conversations])

    def toJSON(self):
        return json.dumps(self, default=lambda o: o.__dict__, sort_keys=True, indent=4)

    @staticmethod
    def fromJSON(path):
        json_dict = json.load(open(path, "r"))
        conversations = []

        for conv in json_dict["conversations"]:
            conversations.append(DataConversation.fromJSON(conv))

        conv_set = ConversationSet(conversations)
        return conv_set
