import inspect
from typing import Any, Callable, Dict, List, Optional, Union

import os
from datetime import datetime
import numpy as np
import torch
from transformers import CLIPTextModel, CLIPTokenizer, T5EncoderModel, T5TokenizerFast
from torchvision.transforms import functional as TF


from diffusers.image_processor import VaeImageProcessor
from diffusers.loaders import FluxLoraLoaderMixin, FromSingleFileMixin, TextualInversionLoaderMixin
from diffusers.models.autoencoders import AutoencoderKL
from diffusers.models.transformers import FluxTransformer2DModel
from diffusers.schedulers import FlowMatchEulerDiscreteScheduler
from diffusers.utils import (
    USE_PEFT_BACKEND,
    is_torch_xla_available,
    logging,
    replace_example_docstring,
    scale_lora_layers,
    unscale_lora_layers,
    BaseOutput,
)
from diffusers.utils.torch_utils import randn_tensor
from diffusers import DiffusionPipeline

from dataclasses import dataclass
from typing import List, Union

import numpy as np
import PIL.Image
from PIL import Image
import torch.nn.functional as F

from diffusers.utils.torch_utils import randn_tensor
from pytorch_wavelets import DWTForward, DWTInverse
from torchvision.transforms import GaussianBlur
import torch.fft as fft
from tqdm import tqdm
from utils import (
    butterworth_low_pass_filter_2d, 
    gaussian_blur_image_sharpening, 
    prep_attn_processor, 
    split_frequency_components_dwt,
    split_frequency_components_fft)

import pdb

if is_torch_xla_available():
    import torch_xla.core.xla_model as xm

    XLA_AVAILABLE = True
else:
    XLA_AVAILABLE = False


logger = logging.get_logger(__name__)  # pylint: disable=invalid-name

EXAMPLE_DOC_STRING = """
    Examples:
        ```py
        >>> import torch
        >>> from diffusers import FluxPipeline

        >>> pipe = FluxPipeline.from_pretrained("black-forest-labs/FLUX.1-schnell", torch_dtype=torch.bfloat16)
        >>> pipe.to("cuda")
        >>> prompt = "A cat holding a sign that says hello world"
        >>> # Depending on the variant being used, the pipeline call will slightly vary.
        >>> # Refer to the pipeline documentation for more details.
        >>> image = pipe(prompt, num_inference_steps=4, guidance_scale=0.0).images[0]
        >>> image.save("flux.png")
        ```
"""

@dataclass
class FluxPipelineOutput(BaseOutput):
    """
    Output class for Stable Diffusion pipelines.

    Args:
        images (`List[PIL.Image.Image]` or `np.ndarray`)
            List of denoised PIL images of length `batch_size` or numpy array of shape `(batch_size, height, width,
            num_channels)`. PIL images or numpy array present the denoised images of the diffusion pipeline.
    """

    images: Union[List[PIL.Image.Image], np.ndarray]


def calculate_shift(
    image_seq_len,
    base_seq_len: int = 256,
    max_seq_len: int = 4096,
    base_shift: float = 0.5,
    max_shift: float = 1.16,
):
    m = (max_shift - base_shift) / (max_seq_len - base_seq_len)
    b = base_shift - m * base_seq_len
    mu = image_seq_len * m + b
    return mu


# Copied from diffusers.pipelines.stable_diffusion.pipeline_stable_diffusion.retrieve_timesteps
def retrieve_timesteps(
    scheduler,
    num_inference_steps: Optional[int] = None,
    device: Optional[Union[str, torch.device]] = None,
    timesteps: Optional[List[int]] = None,
    sigmas: Optional[List[float]] = None,
    **kwargs,
):
    r"""
    Calls the scheduler's `set_timesteps` method and retrieves timesteps from the scheduler after the call. Handles
    custom timesteps. Any kwargs will be supplied to `scheduler.set_timesteps`.

    Args:
        scheduler (`SchedulerMixin`):
            The scheduler to get timesteps from.
        num_inference_steps (`int`):
            The number of diffusion steps used when generating samples with a pre-trained model. If used, `timesteps`
            must be `None`.
        device (`str` or `torch.device`, *optional*):
            The device to which the timesteps should be moved to. If `None`, the timesteps are not moved.
        timesteps (`List[int]`, *optional*):
            Custom timesteps used to override the timestep spacing strategy of the scheduler. If `timesteps` is passed,
            `num_inference_steps` and `sigmas` must be `None`.
        sigmas (`List[float]`, *optional*):
            Custom sigmas used to override the timestep spacing strategy of the scheduler. If `sigmas` is passed,
            `num_inference_steps` and `timesteps` must be `None`.

    Returns:
        `Tuple[torch.Tensor, int]`: A tuple where the first element is the timestep schedule from the scheduler and the
        second element is the number of inference steps.
    """
    if timesteps is not None and sigmas is not None:
        raise ValueError("Only one of `timesteps` or `sigmas` can be passed. Please choose one to set custom values")
    if timesteps is not None:
        accepts_timesteps = "timesteps" in set(inspect.signature(scheduler.set_timesteps).parameters.keys())
        if not accepts_timesteps:
            raise ValueError(
                f"The current scheduler class {scheduler.__class__}'s `set_timesteps` does not support custom"
                f" timestep schedules. Please check whether you are using the correct scheduler."
            )
        scheduler.set_timesteps(timesteps=timesteps, device=device, **kwargs)
        timesteps = scheduler.timesteps
        num_inference_steps = len(timesteps)
    elif sigmas is not None: # enter this branch
        accept_sigmas = "sigmas" in set(inspect.signature(scheduler.set_timesteps).parameters.keys())
        if not accept_sigmas:
            raise ValueError(
                f"The current scheduler class {scheduler.__class__}'s `set_timesteps` does not support custom"
                f" sigmas schedules. Please check whether you are using the correct scheduler."
            )
        scheduler.set_timesteps(sigmas=sigmas, device=device, **kwargs)
        timesteps = scheduler.timesteps
        num_inference_steps = len(timesteps)
    else:
        scheduler.set_timesteps(num_inference_steps, device=device, **kwargs)
        timesteps = scheduler.timesteps
    return timesteps, num_inference_steps


class FluxPipeline(
    DiffusionPipeline,
    FluxLoraLoaderMixin,
    FromSingleFileMixin,
    TextualInversionLoaderMixin,
):
    r"""
    The Flux pipeline for text-to-image generation.


    Args:
        transformer ([`FluxTransformer2DModel`]):
            Conditional Transformer (MMDiT) architecture to denoise the encoded image latents.
        scheduler ([`FlowMatchEulerDiscreteScheduler`]):
            A scheduler to be used in combination with `transformer` to denoise the encoded image latents.
        vae ([`AutoencoderKL`]):
            Variational Auto-Encoder (VAE) Model to encode and decode images to and from latent representations.
        text_encoder ([`CLIPTextModel`]):
            [CLIP], specifically
            the [clip-vit-large-patch14] variant.
        text_encoder_2 ([`T5EncoderModel`]):
            [T5], specifically
            the [google/t5-v1_1-xxl] variant.
        tokenizer (`CLIPTokenizer`):
            Tokenizer of class
            [CLIPTokenizer].
        tokenizer_2 (`T5TokenizerFast`):
            Second Tokenizer of class
            [T5TokenizerFast].
    """

    model_cpu_offload_seq = "text_encoder->text_encoder_2->transformer->vae"
    _optional_components = []
    _callback_tensor_inputs = ["latents", "prompt_embeds"]

    def __init__(
        self,
        scheduler: FlowMatchEulerDiscreteScheduler,
        vae: AutoencoderKL,
        text_encoder: CLIPTextModel,
        tokenizer: CLIPTokenizer,
        text_encoder_2: T5EncoderModel,
        tokenizer_2: T5TokenizerFast,
        transformer: FluxTransformer2DModel,
    ):
        super().__init__()

        self.register_modules(
            vae=vae,
            text_encoder=text_encoder,
            text_encoder_2=text_encoder_2,
            tokenizer=tokenizer,
            tokenizer_2=tokenizer_2,
            transformer=transformer,
            scheduler=scheduler,
        )
        self.vae_scale_factor = (
            2 ** (len(self.vae.config.block_out_channels)) if hasattr(self, "vae") and self.vae is not None else 16
        )
        self.image_processor = VaeImageProcessor(vae_scale_factor=self.vae_scale_factor)
        self.tokenizer_max_length = (
            self.tokenizer.model_max_length if hasattr(self, "tokenizer") and self.tokenizer is not None else 77
        )
        self.default_sample_size = 64

    def _get_t5_prompt_embeds(
        self,
        prompt: Union[str, List[str]] = None,
        num_images_per_prompt: int = 1,
        max_sequence_length: int = 512,
        device: Optional[torch.device] = None,
        dtype: Optional[torch.dtype] = None,
    ):
        device = device or self._execution_device
        dtype = dtype or self.text_encoder.dtype

        prompt = [prompt] if isinstance(prompt, str) else prompt
        batch_size = len(prompt)

        if isinstance(self, TextualInversionLoaderMixin):
            prompt = self.maybe_convert_prompt(prompt, self.tokenizer_2)

        text_inputs = self.tokenizer_2(
            prompt,
            padding="max_length",
            max_length=max_sequence_length,
            truncation=True,
            return_length=False,
            return_overflowing_tokens=False,
            return_tensors="pt",
        )
        text_input_ids = text_inputs.input_ids
        untruncated_ids = self.tokenizer_2(prompt, padding="longest", return_tensors="pt").input_ids

        if untruncated_ids.shape[-1] >= text_input_ids.shape[-1] and not torch.equal(text_input_ids, untruncated_ids):
            removed_text = self.tokenizer_2.batch_decode(untruncated_ids[:, self.tokenizer_max_length - 1 : -1])
            logger.warning(
                "The following part of your input was truncated because `max_sequence_length` is set to "
                f" {max_sequence_length} tokens: {removed_text}"
            )

        prompt_embeds = self.text_encoder_2(text_input_ids.to(device), output_hidden_states=False)[0]

        dtype = self.text_encoder_2.dtype
        prompt_embeds = prompt_embeds.to(dtype=dtype, device=device)

        _, seq_len, _ = prompt_embeds.shape

        # duplicate text embeddings and attention mask for each generation per prompt, using mps friendly method
        prompt_embeds = prompt_embeds.repeat(1, num_images_per_prompt, 1)
        prompt_embeds = prompt_embeds.view(batch_size * num_images_per_prompt, seq_len, -1)

        return prompt_embeds

    def _get_clip_prompt_embeds(
        self,
        prompt: Union[str, List[str]],
        num_images_per_prompt: int = 1,
        device: Optional[torch.device] = None,
    ):
        device = device or self._execution_device

        prompt = [prompt] if isinstance(prompt, str) else prompt
        batch_size = len(prompt)

        if isinstance(self, TextualInversionLoaderMixin):
            prompt = self.maybe_convert_prompt(prompt, self.tokenizer)

        text_inputs = self.tokenizer(
            prompt,
            padding="max_length",
            max_length=self.tokenizer_max_length,
            truncation=True,
            return_overflowing_tokens=False,
            return_length=False,
            return_tensors="pt",
        )

        text_input_ids = text_inputs.input_ids
        untruncated_ids = self.tokenizer(prompt, padding="longest", return_tensors="pt").input_ids
        if untruncated_ids.shape[-1] >= text_input_ids.shape[-1] and not torch.equal(text_input_ids, untruncated_ids):
            removed_text = self.tokenizer.batch_decode(untruncated_ids[:, self.tokenizer_max_length - 1 : -1])
            logger.warning(
                "The following part of your input was truncated because CLIP can only handle sequences up to"
                f" {self.tokenizer_max_length} tokens: {removed_text}"
            )
        prompt_embeds = self.text_encoder(text_input_ids.to(device), output_hidden_states=False)

        # Use pooled output of CLIPTextModel
        prompt_embeds = prompt_embeds.pooler_output
        prompt_embeds = prompt_embeds.to(dtype=self.text_encoder.dtype, device=device)

        # duplicate text embeddings for each generation per prompt, using mps friendly method
        prompt_embeds = prompt_embeds.repeat(1, num_images_per_prompt)
        prompt_embeds = prompt_embeds.view(batch_size * num_images_per_prompt, -1)

        return prompt_embeds

    def encode_prompt(
        self,
        prompt: Union[str, List[str]],
        prompt_2: Union[str, List[str]],
        device: Optional[torch.device] = None,
        num_images_per_prompt: int = 1,
        prompt_embeds: Optional[torch.FloatTensor] = None,
        pooled_prompt_embeds: Optional[torch.FloatTensor] = None,
        max_sequence_length: int = 512,
        lora_scale: Optional[float] = None,
    ):
        r"""

        Args:
            prompt (`str` or `List[str]`, *optional*):
                prompt to be encoded
            prompt_2 (`str` or `List[str]`, *optional*):
                The prompt or prompts to be sent to the `tokenizer_2` and `text_encoder_2`. If not defined, `prompt` is
                used in all text-encoders
            device: (`torch.device`):
                torch device
            num_images_per_prompt (`int`):
                number of images that should be generated per prompt
            prompt_embeds (`torch.FloatTensor`, *optional*):
                Pre-generated text embeddings. Can be used to easily tweak text inputs, *e.g.* prompt weighting. If not
                provided, text embeddings will be generated from `prompt` input argument.
            pooled_prompt_embeds (`torch.FloatTensor`, *optional*):
                Pre-generated pooled text embeddings. Can be used to easily tweak text inputs, *e.g.* prompt weighting.
                If not provided, pooled text embeddings will be generated from `prompt` input argument.
            lora_scale (`float`, *optional*):
                A lora scale that will be applied to all LoRA layers of the text encoder if LoRA layers are loaded.
        """
        device = device or self._execution_device

        # set lora scale so that monkey patched LoRA
        # function of text encoder can correctly access it
        if lora_scale is not None and isinstance(self, FluxLoraLoaderMixin):
            self._lora_scale = lora_scale

            # dynamically adjust the LoRA scale
            if self.text_encoder is not None and USE_PEFT_BACKEND:
                scale_lora_layers(self.text_encoder, lora_scale)
            if self.text_encoder_2 is not None and USE_PEFT_BACKEND:
                scale_lora_layers(self.text_encoder_2, lora_scale)

        prompt = [prompt] if isinstance(prompt, str) else prompt

        if prompt_embeds is None:
            prompt_2 = prompt_2 or prompt
            prompt_2 = [prompt_2] if isinstance(prompt_2, str) else prompt_2

            # We only use the pooled prompt output from the CLIPTextModel
            pooled_prompt_embeds = self._get_clip_prompt_embeds(
                prompt=prompt,
                device=device,
                num_images_per_prompt=num_images_per_prompt,
            )
            prompt_embeds = self._get_t5_prompt_embeds(
                prompt=prompt_2,
                num_images_per_prompt=num_images_per_prompt,
                max_sequence_length=max_sequence_length,
                device=device,
            )

        if self.text_encoder is not None:
            if isinstance(self, FluxLoraLoaderMixin) and USE_PEFT_BACKEND:
                # Retrieve the original scale by scaling back the LoRA layers
                unscale_lora_layers(self.text_encoder, lora_scale)

        if self.text_encoder_2 is not None:
            if isinstance(self, FluxLoraLoaderMixin) and USE_PEFT_BACKEND:
                # Retrieve the original scale by scaling back the LoRA layers
                unscale_lora_layers(self.text_encoder_2, lora_scale)

        dtype = self.text_encoder.dtype if self.text_encoder is not None else self.transformer.dtype
        text_ids = torch.zeros(prompt_embeds.shape[1], 3).to(device=device, dtype=dtype)

        return prompt_embeds, pooled_prompt_embeds, text_ids

    def check_inputs(
        self,
        prompt,
        prompt_2,
        height,
        width,
        prompt_embeds=None,
        pooled_prompt_embeds=None,
        callback_on_step_end_tensor_inputs=None,
        max_sequence_length=None,
    ):
        if height % 8 != 0 or width % 8 != 0:
            raise ValueError(f"`height` and `width` have to be divisible by 8 but are {height} and {width}.")

        if callback_on_step_end_tensor_inputs is not None and not all(
            k in self._callback_tensor_inputs for k in callback_on_step_end_tensor_inputs
        ):
            raise ValueError(
                f"`callback_on_step_end_tensor_inputs` has to be in {self._callback_tensor_inputs}, but found {[k for k in callback_on_step_end_tensor_inputs if k not in self._callback_tensor_inputs]}"
            )

        if prompt is not None and prompt_embeds is not None:
            raise ValueError(
                f"Cannot forward both `prompt`: {prompt} and `prompt_embeds`: {prompt_embeds}. Please make sure to"
                " only forward one of the two."
            )
        elif prompt_2 is not None and prompt_embeds is not None:
            raise ValueError(
                f"Cannot forward both `prompt_2`: {prompt_2} and `prompt_embeds`: {prompt_embeds}. Please make sure to"
                " only forward one of the two."
            )
        elif prompt is None and prompt_embeds is None:
            raise ValueError(
                "Provide either `prompt` or `prompt_embeds`. Cannot leave both `prompt` and `prompt_embeds` undefined."
            )
        elif prompt is not None and (not isinstance(prompt, str) and not isinstance(prompt, list)):
            raise ValueError(f"`prompt` has to be of type `str` or `list` but is {type(prompt)}")
        elif prompt_2 is not None and (not isinstance(prompt_2, str) and not isinstance(prompt_2, list)):
            raise ValueError(f"`prompt_2` has to be of type `str` or `list` but is {type(prompt_2)}")

        if prompt_embeds is not None and pooled_prompt_embeds is None:
            raise ValueError(
                "If `prompt_embeds` are provided, `pooled_prompt_embeds` also have to be passed. Make sure to generate `pooled_prompt_embeds` from the same text encoder that was used to generate `prompt_embeds`."
            )

        if max_sequence_length is not None and max_sequence_length > 512:
            raise ValueError(f"`max_sequence_length` cannot be greater than 512 but is {max_sequence_length}")

    @staticmethod
    def _prepare_latent_image_ids(batch_size, height, width, device, dtype):
        latent_image_ids = torch.zeros(height // 2, width // 2, 3)
        latent_image_ids[..., 1] = latent_image_ids[..., 1] + torch.arange(height // 2)[:, None]
        latent_image_ids[..., 2] = latent_image_ids[..., 2] + torch.arange(width // 2)[None, :]

        latent_image_id_height, latent_image_id_width, latent_image_id_channels = latent_image_ids.shape

        latent_image_ids = latent_image_ids.reshape(
            latent_image_id_height * latent_image_id_width, latent_image_id_channels
        )

        return latent_image_ids.to(device=device, dtype=dtype)

    @staticmethod
    def _pack_latents(latents, batch_size, num_channels_latents, height, width):
        latents = latents.view(batch_size, num_channels_latents, height // 2, 2, width // 2, 2)
        latents = latents.permute(0, 2, 4, 1, 3, 5)
        latents = latents.reshape(batch_size, (height // 2) * (width // 2), num_channels_latents * 4)

        return latents

    @staticmethod
    def _unpack_latents(latents, height, width, vae_scale_factor):
        batch_size, num_patches, channels = latents.shape

        height = height // vae_scale_factor
        width = width // vae_scale_factor

        latents = latents.view(batch_size, height, width, channels // 4, 2, 2)
        latents = latents.permute(0, 3, 1, 4, 2, 5)

        latents = latents.reshape(batch_size, channels // (2 * 2), height * 2, width * 2)

        return latents

    def enable_vae_slicing(self):
        r"""
        Enable sliced VAE decoding. When this option is enabled, the VAE will split the input tensor in slices to
        compute decoding in several steps. This is useful to save some memory and allow larger batch sizes.
        """
        self.vae.enable_slicing()

    def disable_vae_slicing(self):
        r"""
        Disable sliced VAE decoding. If `enable_vae_slicing` was previously enabled, this method will go back to
        computing decoding in one step.
        """
        self.vae.disable_slicing()

    def enable_vae_tiling(self):
        r"""
        Enable tiled VAE decoding. When this option is enabled, the VAE will split the input tensor into tiles to
        compute decoding and encoding in several steps. This is useful for saving a large amount of memory and to allow
        processing larger images.
        """
        self.vae.enable_tiling()

    def disable_vae_tiling(self):
        r"""
        Disable tiled VAE decoding. If `enable_vae_tiling` was previously enabled, this method will go back to
        computing decoding in one step.
        """
        self.vae.disable_tiling()

    def prepare_latents(
        self,
        batch_size,
        num_channels_latents,
        height,
        width,
        dtype,
        device,
        generator,
        latents=None,
    ):
        height = 2 * (int(height) // self.vae_scale_factor)
        width = 2 * (int(width) // self.vae_scale_factor)

        shape = (batch_size, num_channels_latents, height, width)

        if latents is not None:
            latent_image_ids = self._prepare_latent_image_ids(batch_size, height, width, device, dtype)
            return latents.to(device=device, dtype=dtype), latent_image_ids

        if isinstance(generator, list) and len(generator) != batch_size:
            raise ValueError(
                f"You have passed a list of generators of length {len(generator)}, but requested an effective batch"
                f" size of {batch_size}. Make sure the batch size matches the length of the generators."
            )

        latents = randn_tensor(shape, generator=generator, device=device, dtype=dtype)
        latents = self._pack_latents(latents, batch_size, num_channels_latents, height, width)

        latent_image_ids = self._prepare_latent_image_ids(batch_size, height, width, device, dtype)

        return latents, latent_image_ids

    @property
    def guidance_scale(self):
        return self._guidance_scale

    @property
    def joint_attention_kwargs(self):
        return self._joint_attention_kwargs

    @property
    def num_timesteps(self):
        return self._num_timesteps

    @property
    def interrupt(self):
        return self._interrupt

    @torch.no_grad()
    @replace_example_docstring(EXAMPLE_DOC_STRING)
    def __call__(
        self,
        prompt: Union[str, List[str]] = None,
        prompt_2: Optional[Union[str, List[str]]] = None,
        height: Optional[int] = None,
        width: Optional[int] = None,
        num_inference_steps: int = 28,
        timesteps: List[int] = None,
        guidance_scale: float = 3.5,
        num_images_per_prompt: Optional[int] = 1,
        generator: Optional[Union[torch.Generator, List[torch.Generator]]] = None,
        latents: Optional[torch.FloatTensor] = None,
        prompt_embeds: Optional[torch.FloatTensor] = None,
        pooled_prompt_embeds: Optional[torch.FloatTensor] = None,
        output_type: Optional[str] = "pil",
        return_dict: bool = True,
        joint_attention_kwargs: Optional[Dict[str, Any]] = None,
        callback_on_step_end: Optional[Callable[[int, int, Dict], None]] = None,
        callback_on_step_end_tensor_inputs: List[str] = ["latents"],
        max_sequence_length: int = 512,
        #########################
        ntk_factor = None,
        proportional_attention = True,
        text_duplication = True,
        swin_pachify = True,
        #########################
        sharpening_kernel_size = 3,
        sharpening_sigma = (0.1, 2.0),
        sharpening_alpha = 1.0,
        #########################
        target_heights = None,
        target_widths = None,
        num_inference_steps_highres = None,
        filter_ratio = None,
        guidance_scale_highres = None,
        structure_guidance = None,
        upsampling_choice = None,
        alphas = None,
        betas = None,
        flow_choice = None,
    ):
        r"""
        Function invoked when calling the pipeline for generation.

        Args:
            prompt (`str` or `List[str]`, *optional*):
                The prompt or prompts to guide the image generation. If not defined, one has to pass `prompt_embeds`.
                instead.
            prompt_2 (`str` or `List[str]`, *optional*):
                The prompt or prompts to be sent to `tokenizer_2` and `text_encoder_2`. If not defined, `prompt` is
                will be used instead
            height (`int`, *optional*, defaults to self.unet.config.sample_size * self.vae_scale_factor):
                The height in pixels of the generated image. This is set to 1024 by default for the best results.
            width (`int`, *optional*, defaults to self.unet.config.sample_size * self.vae_scale_factor):
                The width in pixels of the generated image. This is set to 1024 by default for the best results.
            num_inference_steps (`int`, *optional*, defaults to 50):
                The number of denoising steps. More denoising steps usually lead to a higher quality image at the
                expense of slower inference.
            timesteps (`List[int]`, *optional*):
                Custom timesteps to use for the denoising process with schedulers which support a `timesteps` argument
                in their `set_timesteps` method. If not defined, the default behavior when `num_inference_steps` is
                passed will be used. Must be in descending order.
            guidance_scale (`float`, *optional*, defaults to 7.0):
                Guidance scale as defined in [Classifier-Free Diffusion Guidance].
                `guidance_scale` is defined as `w` of equation 2. of [Imagen
                Paper]. Guidance scale is enabled by setting `guidance_scale >
                1`. Higher guidance scale encourages to generate images that are closely linked to the text `prompt`,
                usually at the expense of lower image quality.
            num_images_per_prompt (`int`, *optional*, defaults to 1):
                The number of images to generate per prompt.
            generator (`torch.Generator` or `List[torch.Generator]`, *optional*):
                One or a list of [torch generator(s)]
                to make generation deterministic.
            latents (`torch.FloatTensor`, *optional*):
                Pre-generated noisy latents, sampled from a Gaussian distribution, to be used as inputs for image
                generation. Can be used to tweak the same generation with different prompts. If not provided, a latents
                tensor will ge generated by sampling using the supplied random `generator`.
            prompt_embeds (`torch.FloatTensor`, *optional*):
                Pre-generated text embeddings. Can be used to easily tweak text inputs, *e.g.* prompt weighting. If not
                provided, text embeddings will be generated from `prompt` input argument.
            pooled_prompt_embeds (`torch.FloatTensor`, *optional*):
                Pre-generated pooled text embeddings. Can be used to easily tweak text inputs, *e.g.* prompt weighting.
                If not provided, pooled text embeddings will be generated from `prompt` input argument.
            output_type (`str`, *optional*, defaults to `"pil"`):
                The output format of the generate image. Choose between
                [PIL]: `PIL.Image.Image` or `np.array`.
            return_dict (`bool`, *optional*, defaults to `True`):
                Whether or not to return a [`~pipelines.flux.FluxPipelineOutput`] instead of a plain tuple.
            joint_attention_kwargs (`dict`, *optional*):
                A kwargs dictionary that if specified is passed along to the `AttentionProcessor` as defined under
                `self.processor` in
                [diffusers.models.attention_processor].
            callback_on_step_end (`Callable`, *optional*):
                A function that calls at the end of each denoising steps during the inference. The function is called
                with the following arguments: `callback_on_step_end(self: DiffusionPipeline, step: int, timestep: int,
                callback_kwargs: Dict)`. `callback_kwargs` will include a list of all tensors as specified by
                `callback_on_step_end_tensor_inputs`.
            callback_on_step_end_tensor_inputs (`List`, *optional*):
                The list of tensor inputs for the `callback_on_step_end` function. The tensors specified in the list
                will be passed as `callback_kwargs` argument. You will only be able to include variables listed in the
                `._callback_tensor_inputs` attribute of your pipeline class.
            max_sequence_length (`int` defaults to 512): Maximum sequence length to use with the `prompt`.

        Examples:

        Returns:
            [`~pipelines.flux.FluxPipelineOutput`] or `tuple`: [`~pipelines.flux.FluxPipelineOutput`] if `return_dict`
            is True, otherwise a `tuple`. When returning a tuple, the first element is a list with the generated
            images.
        """

        current_time = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
        save_folder = f"./results/{current_time}/"
        if not os.path.exists(save_folder):
            os.makedirs(save_folder)

        with open(save_folder + "config.txt", "w", encoding="utf-8") as file:
            file.write(f"Prompt: {prompt}\n")
            file.write(f"Num_inference_steps: {num_inference_steps_highres}\n")
            file.write(f"Structure_guidance: {structure_guidance}\n")
            file.write(f"FFT Filter Ratio: {filter_ratio}\n")
            file.write(f"Alphas: {alphas}\n")
            file.write(f"Betas: {betas}\n")

        self.save_folder = save_folder
        self.swin_pachify = swin_pachify
        self.structure_guidance = structure_guidance

        height = height or self.default_sample_size * self.vae_scale_factor
        width = width or self.default_sample_size * self.vae_scale_factor

        # 1. Check inputs. Raise error if not correct
        self.check_inputs(
            prompt,
            prompt_2,
            height,
            width,
            prompt_embeds=prompt_embeds,
            pooled_prompt_embeds=pooled_prompt_embeds,
            callback_on_step_end_tensor_inputs=callback_on_step_end_tensor_inputs,
            max_sequence_length=max_sequence_length,
        )

        self._guidance_scale = guidance_scale
        self._joint_attention_kwargs = joint_attention_kwargs
        self._interrupt = False

        # 2. Define call parameters
        if prompt is not None and isinstance(prompt, str):
            batch_size = 1
        elif prompt is not None and isinstance(prompt, list):
            batch_size = len(prompt)
        else:
            batch_size = prompt_embeds.shape[0]

        device = self._execution_device

        lora_scale = (
            self.joint_attention_kwargs.get("scale", None) if self.joint_attention_kwargs is not None else None
        )
        (
            prompt_embeds,
            pooled_prompt_embeds,
            text_ids,
        ) = self.encode_prompt(
            prompt=prompt,
            prompt_2=prompt_2,
            prompt_embeds=prompt_embeds,
            pooled_prompt_embeds=pooled_prompt_embeds,
            device=device,
            num_images_per_prompt=num_images_per_prompt,
            max_sequence_length=max_sequence_length,
            lora_scale=lora_scale,
        )

        # 4. Prepare latent variables
        num_channels_latents = self.transformer.config.in_channels // 4
        latents, latent_image_ids = self.prepare_latents(
            batch_size * num_images_per_prompt,
            num_channels_latents,
            height,
            width,
            prompt_embeds.dtype,
            device,
            generator,
            latents,
        )


        # 5. Prepare timesteps
        sigmas = np.linspace(1.0, 1 / num_inference_steps, num_inference_steps)
        image_seq_len = latents.shape[1] # h * w
        # self.scheduler.config.shift = time_shift_1
        mu = calculate_shift(
            image_seq_len,
            self.scheduler.config.base_image_seq_len,
            self.scheduler.config.max_image_seq_len,
            self.scheduler.config.base_shift,
            self.scheduler.config.max_shift,
        )
        timesteps, num_inference_steps = retrieve_timesteps(
            self.scheduler,
            num_inference_steps, 
            device,
            timesteps, 
            sigmas, 
            mu=mu, 
        )
        num_warmup_steps = max(len(timesteps) - num_inference_steps * self.scheduler.order, 0)
        self._num_timesteps = len(timesteps)


        # handle guidance
        if self.transformer.config.guidance_embeds:
            guidance = torch.full([1], guidance_scale, device=device, dtype=torch.float32)
            guidance = guidance.expand(latents.shape[0])
        else:
            guidance = None

        pred_x0_dict = {}
        height_dict = {}
        width_dict = {}

        # 6. Denoising loop
        with self.progress_bar(total=num_inference_steps) as progress_bar:
            for i, t in enumerate(timesteps):
                if self.interrupt:
                    continue

                # broadcast to batch dimension in a way that's compatible with ONNX/Core ML
                timestep = t.expand(latents.shape[0]).to(latents.dtype)

                noise_pred = self.transformer(
                    hidden_states=latents,
                    timestep=timestep / 1000,
                    guidance=guidance,
                    pooled_projections=pooled_prompt_embeds,
                    encoder_hidden_states=prompt_embeds,
                    txt_ids=text_ids,
                    img_ids=latent_image_ids,
                    joint_attention_kwargs=self.joint_attention_kwargs,
                    return_dict=False,
                    ntk_factor=1.0,
                )[0]

                # compute the previous noisy sample x_t -> x_t-1
                latents_dtype = latents.dtype
                latents, pred_x0 = self.flowmatch_step(noise_pred, t, latents, return_dict=False)

                pred_x0_dict[t.item()] = pred_x0
                height_dict[t.item()] = height
                width_dict[t.item()] = width

                if latents.dtype != latents_dtype:
                    if torch.backends.mps.is_available():
                        # some platforms (eg. apple mps) misbehave due to a pytorch bug:
                        latents = latents.to(latents_dtype)

                if callback_on_step_end is not None:
                    callback_kwargs = {}
                    for k in callback_on_step_end_tensor_inputs:
                        callback_kwargs[k] = locals()[k]
                    callback_outputs = callback_on_step_end(self, i, t, callback_kwargs)

                    latents = callback_outputs.pop("latents", latents)
                    prompt_embeds = callback_outputs.pop("prompt_embeds", prompt_embeds)

                # call the callback, if provided
                if i == len(timesteps) - 1 or ((i + 1) > num_warmup_steps and (i + 1) % self.scheduler.order == 0):
                    progress_bar.update()

                if XLA_AVAILABLE:
                    xm.mark_step()

        prep_attn_processor(self.transformer)
        self.save_image(latents, height, width, save_folder + '1024x1024.png', output_type)
        self.enable_vae_tiling()

        for upscale_step, (target_height, target_width) in enumerate(zip(target_heights, target_widths)):
            
            print(f"### Start Sampling {target_height} x {target_width} Resolution ###")

            guidance_latents = self._unpack_latents(latents, height, width, self.vae_scale_factor)
            guidance_latents = (guidance_latents / self.vae.config.scaling_factor) + self.vae.config.shift_factor
            guidance_image = self.vae.decode(guidance_latents, return_dict=False)[0]
            guidance_image = F.interpolate(guidance_image, (target_height, target_width), mode="bicubic", align_corners=False)
            guidance_image = gaussian_blur_image_sharpening(guidance_image, kernel_size=sharpening_kernel_size, sigma=sharpening_sigma, alpha=sharpening_alpha,)

            guidance_image_save = self.image_processor.postprocess(guidance_image, output_type='pil')
            latents, latent_image_ids = self.encode_vae_latents(guidance_image, batch_size, num_channels_latents, target_height, target_width, )

            
            filter_ratio_list = [filter_ratio[upscale_step] for i in range(num_inference_steps_highres[upscale_step])]
            timesteps, num_inference_steps = retrieve_timesteps(self.scheduler, num_inference_steps, device, timesteps=None, sigmas=sigmas, mu=mu,)
            dlfg_timesteps = self.scheduler.timesteps[-num_inference_steps_highres[upscale_step]:] 

            
            noise = randn_tensor(latents.shape, generator, device=latents.device, dtype=latents.dtype)
            latents = self.scheduler.scale_noise(latents, dlfg_timesteps[None, 0], noise,).to(self.transformer.dtype)
            
            with self.progress_bar(total=num_inference_steps_highres[upscale_step]) as progress_bar:
                for i, t in enumerate(dlfg_timesteps):
            
                    if swin_pachify:
                        if i % 2 == 0: 
                            latents = self._unpack_latents(latents, target_height, target_width, self.vae_scale_factor)
                            latents = F.pad(latents, pad=(1, 1, 1, 1), mode='constant', value=0)
                            latents = self._pack_latents(latents, batch_size, num_channels_latents, target_height // self.vae_scale_factor * 2 + 2, target_width // self.vae_scale_factor * 2 + 2)
                            latent_image_ids = self._prepare_latent_image_ids(batch_size, target_height // self.vae_scale_factor * 2 + 2, target_width // self.vae_scale_factor * 2 + 2, device, latents.dtype)
                            target_height = target_height + 1 * self.vae_scale_factor
                            target_width = target_width + 1 * self.vae_scale_factor
                            self.is_even = True
                        else: 
                            target_height = target_height - 1 * self.vae_scale_factor
                            target_width = target_width - 1 * self.vae_scale_factor
                            latents = self._unpack_latents(latents, target_height + 1 * self.vae_scale_factor, target_width + 1 * self.vae_scale_factor, self.vae_scale_factor)
                            latents = latents[:, :, 1:-1, 1:-1]
                            latents = self._pack_latents(latents, batch_size, num_channels_latents, target_height // self.vae_scale_factor * 2, target_width // self.vae_scale_factor * 2)
                            latent_image_ids = self._prepare_latent_image_ids(batch_size, target_height // self.vae_scale_factor * 2, target_width // self.vae_scale_factor * 2, device, latents.dtype)
                            self.is_even = False

                    filter_shape = [batch_size, num_channels_latents, int(target_height // 8), int(target_width // 8)]
                    freq_filter = butterworth_low_pass_filter_2d(filter_shape, device=self._execution_device, ratio = filter_ratio_list[i])
                    alpha = alphas[upscale_step] * (num_inference_steps_highres[upscale_step] - i) / num_inference_steps_highres[upscale_step] 
                    beta = betas[upscale_step] * (num_inference_steps_highres[upscale_step] - i) / num_inference_steps_highres[upscale_step] 

                    print(f"alpha: {alpha}, beta: {beta}")

                    timestep = t.expand(latents.shape[0]).to(latents.dtype)
                    noise_pred = self.transformer(
                            hidden_states=latents,
                            timestep=timestep / 1000,
                            guidance=torch.tensor([guidance_scale_highres[upscale_step]]).to(self.device), # guidance
                            pooled_projections=pooled_prompt_embeds,
                            encoder_hidden_states=prompt_embeds,
                            txt_ids=text_ids,
                            img_ids=latent_image_ids,
                            joint_attention_kwargs=self.joint_attention_kwargs,
                            return_dict=False,
                            ntk_factor=ntk_factor[upscale_step],
                            text_duplication=text_duplication,
                        )[0]

                    latents_dtype = latents.dtype
                    
                    latents, pred_x0 = self.flowmatch_step(noise_pred, t, latents, return_dict=False, 
                            pred_x0_dict=pred_x0_dict, batch_size = batch_size, num_channels_latents = num_channels_latents, 
                            height_dict = height_dict, width_dict = width_dict, target_height = target_height, target_width = target_width, 
                            upsampling_choice = upsampling_choice, structure_guidance = structure_guidance, freq_filter = freq_filter, 
                            alpha = alpha, beta = beta, flow_choice = flow_choice,)
                
                    latents = latents.to(self.transformer.dtype)
                    pred_x0_dict[t.item()] = pred_x0
                    height_dict[t.item()] = target_height
                    width_dict[t.item()] = target_width

                    if latents.dtype != latents_dtype:
                        if torch.backends.mps.is_available():
                            latents = latents.to(latents_dtype)

                    if callback_on_step_end is not None:
                        callback_kwargs = {}
                        for k in callback_on_step_end_tensor_inputs:
                            callback_kwargs[k] = locals()[k]
                        callback_outputs = callback_on_step_end(self, i, t, callback_kwargs)

                        latents = callback_outputs.pop("latents", latents)
                        prompt_embeds = callback_outputs.pop("prompt_embeds", prompt_embeds)

                    # call the callback, if provided
                    if i == len(timesteps) - 1 or ((i + 1) > num_warmup_steps and (i + 1) % self.scheduler.order == 0):
                        progress_bar.update()

                    if XLA_AVAILABLE:
                        xm.mark_step()
            
            self.save_image(latents, target_height, target_width, save_folder + f'{target_height}x{target_width}_n{num_inference_steps_highres[upscale_step]}_f{filter_ratio[upscale_step]}_a{alphas[upscale_step]}_b{betas[upscale_step]}.png', output_type)
            height, width = target_height, target_width

        if output_type == "latent":
            image = latents
        else:
            latents = self._unpack_latents(latents, target_height, target_width, self.vae_scale_factor)
            latents = (latents / self.vae.config.scaling_factor) + self.vae.config.shift_factor
            image = self.vae.decode(latents, return_dict=False)[0]
            image = self.image_processor.postprocess(image, output_type=output_type)

        # Offload all models
        self.maybe_free_model_hooks()

        if not return_dict:
            return (image,)

        print(f"### All Done! Saved in {save_folder} ###")

        return FluxPipelineOutput(images=image)

    def save_image(self, latents, height, width, output_path, output_type, ):
        latents = self._unpack_latents(latents, height, width, self.vae_scale_factor)
        latents = (latents / self.vae.config.scaling_factor) + self.vae.config.shift_factor
        image = self.vae.decode(latents.to(self.vae.dtype), return_dict=False)[0]
        image = self.image_processor.postprocess(image, output_type=output_type)
        image[0].save(output_path)
        return 

    def decode_vae_latents(self, latents, height, width, output_type, ):
        latents = self._unpack_latents(latents, height, width, self.vae_scale_factor)
        latents = (latents / self.vae.config.scaling_factor) + self.vae.config.shift_factor
        image = self.vae.decode(latents.to(self.vae.dtype), return_dict=False)[0]
        image = self.image_processor.postprocess(image, output_type=output_type)
        return image

    def encode_vae_latents(self, image, batch_size, num_channels_latents, height, width,):
        height = 2 * (int(height) // self.vae_scale_factor)
        width = 2 * (int(width) // self.vae_scale_factor)
        latents = self.vae.encode(image.to(self.vae.dtype).to(self.vae.device)).latent_dist.mode()
        latents = (latents - self.vae.config.shift_factor) * self.vae.config.scaling_factor
        latents = self._pack_latents(latents, batch_size, num_channels_latents, height, width)
        latent_image_ids = self._prepare_latent_image_ids(batch_size, height, width, self.vae.device, self.vae.dtype)
        return latents, latent_image_ids
        
    def split_freq(self, x, freq_filter, is_low = True):
        x_freq = fft.fftshift(fft.fft2(x.to(freq_filter.dtype)))
        x_split_freq = x_freq * freq_filter if is_low else x_freq * (1 - freq_filter)
        x_split = fft.ifft2(fft.ifftshift(x_split_freq)).real
        return x_split

    def flowmatch_step(
        self,
        model_output: torch.FloatTensor,
        timestep: Union[float, torch.FloatTensor],
        sample: torch.FloatTensor,
        s_churn: float = 0.0,
        s_tmin: float = 0.0,
        s_tmax: float = float("inf"),
        s_noise: float = 1.0,
        generator: Optional[torch.Generator] = None,
        return_dict: bool = True,
        pred_x0_dict = None,
        batch_size = None,
        num_channels_latents = None,
        height_dict = None,
        width_dict = None,
        target_height = None,
        target_width = None,
        upsampling_choice = None,
        structure_guidance = None,
        freq_filter = None, 
        alpha = None,
        beta = None,
        flow_choice = None,
        ):
        
        if self.scheduler.step_index is None:
            self.scheduler._init_step_index(timestep)
            self.model_output_high = None
            self.model_output_ref = None

        # Upcast to avoid precision issues when computing prev_sample
        sample = sample.to(torch.float32)

        sigma = self.scheduler.sigmas[self.scheduler.step_index]
        sigma_next = self.scheduler.sigmas[self.scheduler.step_index + 1]

        pred_x0 = sample - model_output * sigma
        pred_x1 = sample + model_output * (1 - sigma)
        original_pred_x0 = pred_x0

        if flow_choice is not None: 

            pred_x0 = self._unpack_latents(pred_x0, target_height, target_width, self.vae_scale_factor)
            pred_x0_l = self._unpack_latents(pred_x0_dict[timestep.item()], height_dict[timestep.item()], width_dict[timestep.item()], self.vae_scale_factor) # 98.20803833007812
            
            if upsampling_choice == "latent":
                pred_x0_ref = F.interpolate(pred_x0_l, (int(target_height // 8), int(target_width // 8)), mode="bicubic", align_corners=False)
            elif upsampling_choice == "pixel":
                pred_x0_l = (pred_x0_l / self.vae.config.scaling_factor) + self.vae.config.shift_factor
                pred_x0_l_image = self.vae.decode(pred_x0_l.to(self.vae.dtype), return_dict=False)[0]
                pred_x0_l_image = F.interpolate(pred_x0_l_image, (target_height, target_width), mode="bicubic", align_corners=False)
                pred_x0_l_image = gaussian_blur_image_sharpening(pred_x0_l_image, kernel_size=3, sigma=(0.1, 2.0), alpha=1.0,)
                pred_x0_ref, _ = self.encode_vae_latents(pred_x0_l_image, batch_size, num_channels_latents, target_height, target_width, )
                pred_x0_ref = self._unpack_latents(pred_x0_ref, target_height, target_width, self.vae_scale_factor)
            else:
                assert False

            if structure_guidance == "fft":
                pred_x0 = pred_x0 + alpha * (split_frequency_components_fft(pred_x0_ref, freq_filter, is_low = True) - split_frequency_components_fft(pred_x0, freq_filter, is_low = True))

            elif structure_guidance == "dwt":
                pred_x0 = pred_x0 + alpha * (split_frequency_components_dwt(pred_x0_ref, level=1) - split_frequency_components_dwt(pred_x0, level=1))
            
            else:
                assert False

            pred_x0 = self._pack_latents(pred_x0, batch_size, num_channels_latents, 2 * (int(target_height) // self.vae_scale_factor), 2 * (int(target_width) // self.vae_scale_factor))
            pred_x0_ref = self._pack_latents(pred_x0_ref, batch_size, num_channels_latents, 2 * (int(target_height) // self.vae_scale_factor), 2 * (int(target_width) // self.vae_scale_factor))
            
            model_output = (sample - pred_x0) / (sigma + 1e-6) 
            model_output_ref = (sample - pred_x0_ref) / (sigma + 1e-6) 
            if self.model_output_high is not None:
                if self.swin_pachify:
                    if self.is_even: 
                        self.model_output_high = self._unpack_latents(self.model_output_high, target_height - 1 * self.vae_scale_factor, target_width - 1 * self.vae_scale_factor, self.vae_scale_factor)
                        self.model_output_high = F.pad(self.model_output_high, pad=(1, 1, 1, 1), mode='constant', value=0)
                        self.model_output_high = self._pack_latents(self.model_output_high, batch_size, num_channels_latents, target_height // self.vae_scale_factor * 2, target_width // self.vae_scale_factor * 2)
                        self.model_output_ref = self._unpack_latents(self.model_output_ref, target_height - 1 * self.vae_scale_factor, target_width - 1 * self.vae_scale_factor, self.vae_scale_factor)
                        self.model_output_ref = F.pad(self.model_output_ref, pad=(1, 1, 1, 1), mode='constant', value=0)
                        self.model_output_ref = self._pack_latents(self.model_output_ref, batch_size, num_channels_latents, target_height // self.vae_scale_factor * 2, target_width // self.vae_scale_factor * 2)
                    else: 
                        self.model_output_high = self._unpack_latents(self.model_output_high, target_height + 1 * self.vae_scale_factor, target_width + 1 * self.vae_scale_factor, self.vae_scale_factor)
                        self.model_output_high = self.model_output_high[:, :, 1:-1, 1:-1]
                        self.model_output_high = self._pack_latents(self.model_output_high, batch_size, num_channels_latents, target_height // self.vae_scale_factor * 2, target_width // self.vae_scale_factor * 2)
                        self.model_output_ref = self._unpack_latents(self.model_output_ref, target_height + 1 * self.vae_scale_factor, target_width + 1 * self.vae_scale_factor, self.vae_scale_factor)
                        self.model_output_ref = self.model_output_ref[:, :, 1:-1, 1:-1]
                        self.model_output_ref = self._pack_latents(self.model_output_ref, batch_size, num_channels_latents, target_height // self.vae_scale_factor * 2, target_width // self.vae_scale_factor * 2)

                model_output = model_output + beta * (self.model_output_high + model_output_ref - self.model_output_ref - model_output) 
            self.model_output_high = model_output
            self.model_output_ref = model_output_ref

        prev_sample = sample + (sigma_next - sigma) * model_output
        
        # Cast sample back to model compatible dtype
        prev_sample = prev_sample.to(model_output.dtype)

        # upon completion increase step index by one
        self.scheduler._step_index += 1
        
        return (prev_sample, original_pred_x0)



