import json
import argparse
import os
import copy
import torch
from datasets import load_dataset
from transformers import AutoModelForCausalLM, AutoTokenizer, AutoConfig

from experiment.experiment import ExperimentAlign

from utils.utils import timestamp, get_remove_mlp, get_remove_mha, get_remove_ln


"""
Parse arguments passed to script.
"""
parser = argparse.ArgumentParser("run config runner for metrics across prompts")
parser.add_argument("--data", action="store", type=str)
parser.add_argument("--model", action="store", type=str)
parser.add_argument("--untrained-model", action="store", type=str)
parser.add_argument("--config", action="store", type=str)
parser.add_argument("--start", action="store", type=int)
parser.add_argument("--end", action="store", type=int)
parser.add_argument("--quantize", action="store_false", default=True) # Quantize model when loading?
parser.add_argument("--untrained", action="store_true", default=False) # Load untrained model?
parser.add_argument("--remove", action="store_true", default=False) # arg to remove pieces of architecture

args = parser.parse_args()

# device = torch.device("cuda:0")


"""
Set up model and data configs.
"""

config = json.load(open(args.config, "r"))
experiment_files = os.listdir(experiment_dir)


"""
Load model from specified path.
"""
print('Memory info:', torch.cuda.mem_get_info())
model_name = args.model
model_path = config["MODEL_PATH"][model_name]
timestamp(f"Loading {model_name} from {model_path}")
model = AutoModelForCausalLM.from_pretrained(
    model_path,
    device_map="cuda",
    trust_remote_code=True,
    load_in_4bit=args.quantize
)
#print(model.device)
timestamp(f"{model_name} has been successfully loaded from {model_path}")
print('Memory info:', torch.cuda.mem_get_info())


timestamp("Loading tokenizer from " + model_path)
tokenizer = AutoTokenizer.from_pretrained(
    model_path,
    use_fast=True
)
#print(tokenizer.device)
timestamp(f"{model_name} tokenizer has been successfully loaded from {model_path}")

models = [model]
model_types = ["standard"]


if args.untrained:
    timestamp(f"Loading untrained {model_name}")
    config_untrained = AutoConfig.from_pretrained(model_path)
    untrained_model =  AutoModelForCausalLM.from_config(config_untrained)
    timestamp(f"Untrained {model_name} has been successfully loaded")
    models.append(untrained_model)
    model_types.append("untrained")

data_name = args.data
data_config = config["DATA_CONFIG"][data_name]
timestamp(f"Running over the {data_name} dataset")

try:
    dataset = load_dataset(*data_config.values())
    timestamp("Loaded " + data_name + " from cache.")

    """
    Run experiment.
    """
    experiment = ExperimentAlign(
        models,
        model_types,
        tokenizer,
        model_name,
        alignment = True,
        out_dir=experiment_dir
    )
    # timestamp("Created experiment " + str(config["EXPERIMENT_CONFIG"]))

    outputs, out_file = experiment.run(
        dataset=dataset,
        data_name=data_name,
        start=args.start,
        end=args.end,
        save=True
    )
    timestamp(f"Finished experiment on {model_name} over {data_name} dataset and saved in  " + out_file)

except Exception as e:
    print(f"An error occurred: {e}")
