
%The main function. Want to solve min_x ||Ax-b||_p subject to constraints
%B'*x = d to a (1+eps) approximation.

% For notation: m is the dimension, n is the number of constraints
function [final,call]= IRLS(eps,p,B,d,A,b)
    [n, m] = size(A);
    x = InitialSoln(B, d, A, b);

    lb = norm(A*x-b, 2)/n^(1/2-1/p);
    [final,call] = pNorm_with_initial_vector(eps,p,B,d,A,b,x,lb);
    fprintf("Iteration: %d\n:" ,call);
end

% The main algorithm.
function [final_vec,call] = pNorm_with_initial_vector(eps,p,B,d,A,b,x,lb)
    % lb is a lower bound on the objective
    % Initial Solution
    current = norm(A*x-b,p); 
    % Check if the initial solution is 0. In that case return 0.
    if current^p ==0					                             
        disp("Norm = 0");
        % println("Max gradient entry:", max_gradient_entry);
    	call = 1;
        final_vec = x;
        return
    end
    call = 0;
    [n,m] = size(A);

    
    % Initial padding. An upper bound on (Initial_Solution - OPT)/16p.
    i = (current^p - lb^p)/(16*p)	; 
    eps_2 = 1;
    
    % Termination condition, if this is achieved, we have a (1+eps)-approximate solution
    while i > 2*eps*current^p/(16*p*(1+eps)) 

        [E, delta, cnt] = LpSolveReg(p,B,d,A,b,i,x,eps_2);
        call = call + cnt;


        delta = delta/p;

        % Find the step size that gives the minimum objective given the step delta.
        % alpha = LineSearchObj(A,b,x,p,delta);  

        alpha = LineSearchObj(A,b,x,p,delta);  
        
        need_to_reduce_i = false;
        
        % Check if we have had sufficient progress, if not reduce the padding
        % We add the condition if E >  2*i^(2/p), ie, the L2 solver is infeasible, 
        % we also reduce i
        if E >  2*i^(2/p) || 2*p*dot((abs(A*x-b)).^(p-2), (A*delta).^2) > 2*i             
            need_to_reduce_i = true;
        end
        % Check if our new norm is less than the current norm, and then
        % update x. It is possible to get a value that does not reduce the
        % norm because the line search does not solve to very high
        % precision.
        if norm(A*(x-alpha*delta)-b,p) < current        
            x = x-alpha*delta;
            current = norm(A*x-b,p);
            fprintf('Reducing norm: %d \n', current^p);
        else
            % If we do not reduce the norm, we reduce the padding i.
            need_to_reduce_i = true;                    
        end
        if need_to_reduce_i
            i = i/2;
            fprintf('Reducing i: %d \n', i);
        end
        i = min(i, (current^p - lb^p)/(16*p)); 
    end
    final_vec = x;
end



function [ E, f, cnt ] = LpSolveReg(p,B,d,A,b,i,x,eps)
    %Solve min ||A*f||_p : B' * f = d
    %   output (1+eps) approximate solution consisting of
    %          flow f and resistances r

    s = i^((2-p)/p);
    r = 2*(abs(A*x-b)).^(p-2);
    g = r .*(A*x-b);

    gamma = s*r;

    OPT = 2*i^(1/p); 


    
    [n, m] = size(A);
    q = p/(p-2);
    r = ones(n, 1); r = r / norm(r, q) / 2^(1/q);

    [f, E] = electric(B, r+gamma, d, g, i, A);
    E = E/max(1, norm(r, q));

    

    cnt = 1;
    while E * (1+eps) < OPT^2 && norm(r, q) < 1
        alpha = next_alpha(q, r, A*f, E, OPT, eps);
        if (sum(alpha-1) > 1)
            r = r.*alpha;
            [f, E] = electric(B, r+gamma, d, g, i, A);
            E = E/max(1, norm(r, q));
        else
            break;
        end
        cnt = cnt + 1; 
        
    end
    
    % fprintf('Lq solver finished after %d iterations\n', cnt);
end
    
function [ alpha ] = next_alpha(q, r, f, E, OPT, eps)
    n = size(r, 1);
    gamma = norm(r,q)^(q-1)/OPT^2*f.^2./r.^(q-1);
    alpha = ones(n,1);
    for j=1:size(alpha,1)
        if gamma(j) >= (1+eps)
            alpha(j) = gamma(j)^(1/q);
        end
    end
end


function [ f ] = InitialSoln(B, d, A, b)
    

    Calc = transpose(A)*A;
    t = transpose(A) * b;

    if max(abs(B)) == 0
        f = Calc \ t;
    else
        S = transpose(B) / Calc;
        L = S * B;
        o = d - S * t;
        v = L\ o;
        z = B * v + t;
        f = Calc \ z;
    end
end


function [ f, E ] = electric(B, r, d, g, i, A)
    
    [n, m] = size(A);

    if max(abs(B)) == 0
        
        R = spdiags(r, zeros(1,1), n, n);
        Calc = transpose(A)*R*A;
        inverseCalc = Calc \ (transpose(A)*g);
        quadform = transpose(g)*A*inverseCalc;
        f = i*inverseCalc/(2*quadform) ; 
        E = transpose(f)*Calc*f;

    else
        R = spdiags(r, zeros(1,1), n, n);
        B_aug = transpose([transpose(B);transpose(g)*A]);
        d_aug = [zeros(size(B,2),1);i/2];

        Calc = transpose(A)*R*A;
        % inverseCalc = inv(Calc);
        S = transpose(B_aug) / Calc;
        L = S * B_aug;
        o = d_aug;
        % v = cgsolve(L, o, 1e-12, 10000, 0);
        v = L \ o;
        z = B_aug * v;
        f = Calc \ z;
        E = transpose(f)*Calc*f;
    end


end


% A function that calculates the gradient of ||A(x-scale*delta)-b||_p^p.
% Here A,b are as in the input. We use this in the next function to find a
% scale so that given the current solution x and the next step delta, we
% can scale delta so as to make maximum progress.
function obj = GradientScaledObj(scale,p,z,w)
    v = z - scale*w;
    y = abs(v).^(p-2);
    y1 = v .* y;
    obj = -1 * (w' * y1);
end

% This finds a scaling so that given the current solution x and the next
% step delta, we can scale delta so as to make maximum progress.
function alpha = LineSearchObj(A,b,x,p,delta)
    L = -3;
    U = 3;
    w = A * delta;
    z = A * x - b;
    while GradientScaledObj(U,p,z,w)<0
        L = U;
        U = 2*U;
    end
    while GradientScaledObj(L,p,z,w)>0
        U = L;
        L = 2*L;
    end
    assert (GradientScaledObj(L,p,z,w) < 0);
    assert (GradientScaledObj(U,p,z,w) > 0);
    while abs(U-L)>1e-1
        if (GradientScaledObj((L+U)/2,p,z,w)>0)
            U = (L+U)/2;
        else
            L = (L+U)/2;
        end
    end
    alpha = (L+U)/2;

end