from typing import Text
import numpy as np
import scipy as sp
from hashlib import md5
import json

# scipy libraries need separate import
import scipy.spatial


def cluster(arr, tol=1e-5, metric='euclidean'):
  dim = arr.shape[-1]

  dist = sp.spatial.distance.pdist(arr, metric=metric) / (dim ** 0.5)
  dist = sp.spatial.distance.squareform(dist)

  mask = np.unique(dist < tol, axis=-1).T
  return np.array([arr[m].mean(0) for m in mask])


def recursive_cluster(arr, max_iters=10, tol=1e-5, metric='euclidean'):
  n = len(arr)

  for _ in range(max_iters):
    arr = cluster(arr, tol=tol, metric=metric)
    if len(arr) == n:
      break
    else:
      n = len(arr)

  return arr


def get_cluster_ids(arr, clusters, metric='euclidean'):
  return sp.spatial.distance.cdist(arr, clusters, metric=metric).argmin(-1)


def utility_jumps(utility, strategies, clusters=None, cluster_ids=None):
  current_util = utility(strategies)
  if clusters is None:
    clusters = recursive_cluster(strategies)
  if cluster_ids is None:
    cluster_ids = get_cluster_ids(strategies, clusters)

  diffs = {}
  for cluster_id in range(len(clusters)):
    diffs[cluster_id] = []
    pid = np.argmax(cluster_ids == cluster_id)  # 1st producer from the cluster
    for alt_cluster in clusters:
      s_alt = strategies.at[pid].set(alt_cluster)
      alt_util = utility(s_alt)[pid]
      diff = alt_util - current_util[pid]

      diffs[cluster_id].append(diff)  # optionally can break at first large diff

  return {k: np.array(v) for k, v in diffs.items()}


def get_run_id(value_dict) -> Text:
  """Get hash of value dictionary for ID purposes."""
  return md5(json.dumps(value_dict, sort_keys=True).encode('utf-8')).hexdigest()
