from collections import OrderedDict

import torch
import torch.nn as nn
import torch.nn.functional as F


class BasicBlock(nn.Module):
    def __init__(self, in_planes, out_planes, stride, dropRate=0.0):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv2 = nn.Conv2d(out_planes, out_planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_planes)
        self.relu = nn.ReLU(inplace=True)
        self.droprate = dropRate
        self.equalInOut = (in_planes == out_planes)
        self.convShortcut = (not self.equalInOut) and nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, padding=0, bias=False) or None

    def forward(self, x):
        if not self.equalInOut:
            x = self.relu(self.bn1(x))
        else:
            out = self.relu(self.bn1(x))

        out = self.relu(self.bn2(self.conv1(out if self.equalInOut else x)))

        if self.droprate > 0:
            out = F.dropout(out, p=self.droprate, training=self.training)

        out = self.conv2(out)
        return torch.add(x if self.equalInOut else self.convShortcut(x), out)


class WideResNet(nn.Module):
    def __init__(self, num_classes, depth, widen_factor=1, dropRate=0.0):
        super(WideResNet, self).__init__()
        assert (depth - 4) % 6 == 0
        n = (depth - 4) // 6
        num_channels = [16, 16*widen_factor, 32*widen_factor, 64*widen_factor]
        self.channels = num_channels[0]

        # 1st conv before any network block
        self.conv = nn.Conv2d(3, num_channels[0], kernel_size=3, stride=1, padding=1, bias=False)

        self.block1 = self._make_block(BasicBlock, num_channels[1], n, 1, dropRate)
        self.block2 = self._make_block(BasicBlock, num_channels[2], n, 2, dropRate)
        self.block3 = self._make_block(BasicBlock, num_channels[3], n, 2, dropRate)

        self.bn = nn.BatchNorm2d(num_channels[3])
        self.relu = nn.ReLU(inplace=True)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.flatten = nn.Flatten(start_dim=1)
        self.fc = nn.Linear(num_channels[3], num_classes)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.Linear):
                m.bias.data.zero_()

    def _make_block(self, block, out_channels, nb_layers, stride, dropRate):
        layers = OrderedDict([("layer0", block(self.channels, out_channels, stride, dropRate))])
        for i in range(1, nb_layers):
            layers[f"layer{i}"] = block(out_channels, out_channels, 1, dropRate)
        self.channels = out_channels

        return nn.Sequential(layers)

    def forward(self, x):
        out = self.conv(x)
        out = self.block1(out)
        out = self.block2(out)
        out = self.block3(out)
        out = self.relu(self.bn(out))
        out = self.avgpool(out)
        out = self.flatten(out)
        out = self.fc(out)
        return out

def WideResNet_28_10(num_classes):
    return WideResNet(num_classes, 28, 10)

def WideResNet_34_20(num_classes):
    return WideResNet(num_classes, 34, 20)

# unit test
if __name__ == "__main__":
    torch.manual_seed(1)
    images = torch.rand(4, 3, 32, 32)
    net = WideResNet(10, 28, 10)
    net.eval()
    with torch.no_grad():
        assert  round(net(images).sum().item(), 3) == 1.575