# -*- coding: utf-8 -*-
# @Author: Chen Renjie
# @Date:   2021-08-16 23:50:51
# @Last Modified by:   Chen Renjie
# @Last Modified time: 2021-08-17 00:49:56

from collections import OrderedDict
import torch
import torch.nn as nn
import torch.nn.functional as F

class PreActBlock(nn.Module):
    '''Pre-activation version of the BasicBlock.'''
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(PreActBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)

        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False)
            )

    def forward(self, x):
        out = F.relu(self.bn1(x))
        shortcut = self.shortcut(out) if hasattr(self, "shortcut") else x
        out = self.conv1(out)
        out = self.conv2(F.relu(self.bn2(out)))
        out += shortcut
        return out


class PreActResNet(nn.Module):

    def __init__(self, block, num_blocks, num_classes=10):
        super(PreActResNet, self).__init__()
        self.in_planes = 64

        self.other_layers = nn.ModuleList()

        self.first_layer = nn.Conv2d(3, 64, kernel_size=3, stride=1, padding=1, bias=False)


        self.other_layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.other_layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.other_layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.other_layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)

        self.linear = GlobalpoolFC(512 * block.expansion, num_classes)
        self.other_layers.append(self.linear)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.other_layers.append(layers[-1])

            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):

        x = self.first_layer(x)
        self.first_layer_out = x
        self.first_layer_out.requires_grad_()
        self.first_layer_out.retain_grad()
        # x = self.layer_one_out

        for layer in self.other_layers:
            x = layer(x)
        return x

class GlobalpoolFC(nn.Module):

    def __init__(self, num_in, num_class):
        super(GlobalpoolFC, self).__init__()
        self.pool = nn.AdaptiveAvgPool2d(output_size=1)
        self.fc = nn.Linear(num_in, num_class)

    def forward(self, x):
        y = self.pool(x)
        y = y.reshape(y.shape[0], -1)
        y = self.fc(y)
        return y


class PreActBottleneck(nn.Module):
    '''Pre-activation version of the original Bottleneck module.'''
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(PreActBottleneck, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn3 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion*planes, kernel_size=1, bias=False)

        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False)
            )

    def forward(self, x):
        out = F.relu(self.bn1(x))
        shortcut = self.shortcut(out) if hasattr(self, 'shortcut') else x
        out = self.conv1(out)
        out = self.conv2(F.relu(self.bn2(out)))
        out = self.conv3(F.relu(self.bn3(out)))
        out += shortcut
        return out


def PreActResNet18_yopo(num_classes):
    return PreActResNet(PreActBlock, [2, 2, 2, 2], num_classes)

def PreActResNet34_yopo(num_classes):
    return PreActResNet(PreActBlock, [3, 4, 6, 3], num_classes)



class BasicBlock(nn.Module):
    expansion = 1
    def __init__(self, in_channels, out_channels, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(in_channels, out_channels, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(out_channels)
        self.conv2 = nn.Conv2d(out_channels, out_channels, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_channels)
        self.relu = nn.ReLU(inplace=True)

        # shortcut down sample
        if stride != 1 or in_channels != self.expansion*out_channels:
            self.shortcut = nn.Sequential(OrderedDict([
                ("conv", nn.Conv2d(in_channels, self.expansion*out_channels, kernel_size=1, stride=stride, padding=0, bias=False)),
                ("bn", nn.BatchNorm2d(self.expansion*out_channels))
            ]))
        else:
            self.shortcut = nn.Sequential()

    def forward(self, x):
        identity = self.shortcut(x)

        out = self.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))

        out = out + identity
        out = self.relu(out)

        return out

class ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes):
        super(ResNet, self).__init__()
        self.hooks = {}
        self.handles = {}
        self.channels = 64
        self.other_layers = nn.ModuleList()
        self.first_layer = nn.Conv2d(in_channels=3, out_channels=64, kernel_size=3, stride=1, padding=1, bias=False)
        self.layer0 = nn.Sequential(OrderedDict([
            ("bn", nn.BatchNorm2d(num_features=64)),
            ("relu", nn.ReLU(inplace=True))
        ]))

        self.layer1 = self.make_layer(block,  64, num_blocks[0], 1)
        self.layer2 = self.make_layer(block, 128, num_blocks[1], 2)
        self.layer3 = self.make_layer(block, 256, num_blocks[2], 2)
        self.layer4 = self.make_layer(block, 512, num_blocks[3], 2)

        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.flatten = nn.Flatten(start_dim=1)
        self.fc = nn.Linear(512*block.expansion, num_classes)

        self.other_layers.append(self.layer0)
        self.other_layers.append(self.layer1)
        self.other_layers.append(self.layer2)
        self.other_layers.append(self.layer3)
        self.other_layers.append(self.layer4)
        self.other_layers.append(self.avgpool)
        self.other_layers.append(self.flatten)
        self.other_layers.append(self.fc)

    def make_layer(self, block, out_channels, block_num, stride):
        layers = OrderedDict([("block0", block(self.channels, out_channels, stride))])
        self.channels = out_channels * block.expansion
        for i in range(block_num - 1):
            layers[f"block{i+1}"] = block(self.channels, out_channels, 1)
            self.channels = out_channels * block.expansion

        return nn.Sequential(layers)


    def forward(self, x):
        x = self.first_layer(x)
        self.first_layer_out = x
        self.first_layer_out.requires_grad_()
        self.first_layer_out.retain_grad()

        for layer in self.other_layers:
            x = layer(x)
        return x


def ResNet18_yopo(num_classes):
    return ResNet(BasicBlock, [2, 2, 2, 2], num_classes)



