# -*- coding: utf-8 -*-
from __future__ import division
import numpy as np
import torch
import random
from torch import optim
from torch.nn.utils import clip_grad_norm_
from model import DQN


 
class Agent():
  def __init__(self, args, env, seed):
    random.seed(seed)

    self.atoms = args.atoms
    self.Vmin = args.V_min
    self.Vmax = args.V_max
    self.support = torch.linspace(args.V_min, args.V_max, self.atoms).to(device=args.device)  # Support (range) of z
    self.delta_z = (args.V_max - args.V_min) / (self.atoms - 1)
    self.batch_size = args.batch_size
    self.n = args.multi_step
    self.discount = args.discount
    self.norm_clip = args.norm_clip
    self.learning  = args.learning_rate
    self.env       = env
    self.num_action= env.action_space.n
    self.epsilon   = args.start_epsilon
    
    self.online_net = DQN(args, self.num_action).to(device=args.device)
    self.target_net = DQN(args, self.num_action).to(device=args.device)
    self.update_target_net()
    self.target_net.train()
    for param in self.target_net.parameters():
      param.requires_grad = False
    
    self.optimiser = optim.Adam(self.online_net.parameters(), lr=args.learning_rate, eps=args.adam_eps)
  
  # Resets noisy weights in all linear layers (of online net only)
  def reset_noise(self):
    self.online_net.reset_noise()

  # Acts based on single state (no batch)
  # def act(self, state):
  #   with torch.no_grad():
  #     q_values = (self.online_net(state.unsqueeze(0))  * self.support ).sum(2)
  #     prob     = torch.exp(q_values) / torch.exp(q_values).sum()
  #     prob     = prob.cpu().numpy().flatten()
  #     action   = np.random.choice(self.num_action, 1, p=prob)

  #     # action = (self.online_net(state.unsqueeze(0)) * self.support).sum(2).argmax(1).cpu().numpy()
  #     return action
  
  def act(self, state,probs_llm):
    if np.random.random() < self.epsilon:
      if probs_llm != None and len(probs_llm)==self.num_action:
          actions = [q for q in range(0,self.num_action)]
          action = random.choices(actions, weights=probs_llm, k=1)[0]
      else:
          action = self.env.action_space.sample() # sample
      action_flag = 1
    else:
      with torch.no_grad():
        # q_values = (self.online_net(state.unsqueeze(0))  * self.support ).sum(2)
        # prob     = torch.exp(q_values) / torch.exp(q_values).sum()
        # prob     = prob.cpu().numpy().flatten()
        # action   = np.random.choice(self.num_action, 1, p=prob)
        action = (self.online_net(state.unsqueeze(0)) * self.support).sum(2).argmax(1).cpu().numpy().item()
      action_flag = 0

    return action, action_flag

  def learn(self, mem):
    # Sample transitions
    idxs, states, actions, returns, next_states, nonterminals, weights = mem.sample(self.batch_size)

    # Calculate current state probabilities (online network noise already sampled)
    log_ps = self.online_net(states, log=True)  # Log probabilities log p(s_t, ·; θonline)

    log_ps_a = log_ps[range(self.batch_size), actions]  # log p(s_t, a_t; θonline)

    # Fixed weight (No learning)
    with torch.no_grad():
      # Calculate nth next state probabilities
      pns = self.online_net(next_states)  # Probabilities p(s_t+n, ·; θonline)
      dns = self.support.expand_as(pns) * pns  # Distribution d_t+n = (z, p(s_t+n, ·; θonline))
      argmax_indices_ns = dns.sum(2).argmax(1)  # Perform argmax action selection using online network: argmax_a[(z, p(s_t+n, a; θonline))]
      self.target_net.reset_noise()  # Sample new target net noise
      pns = self.target_net(next_states)  # Probabilities p(s_t+n, ·; θtarget)
      pns_a = pns[range(self.batch_size), argmax_indices_ns]  # Double-Q probabilities p(s_t+n, argmax_a[(z, p(s_t+n, a; θonline))]; θtarget)

      # Compute Tz (Bellman operator T applied to z)
      Tz = returns.unsqueeze(1) + nonterminals * (self.discount ** self.n) * self.support.unsqueeze(0)  # Tz = R^n + (γ^n)z (accounting for terminal states)
      Tz = Tz.clamp(min=self.Vmin, max=self.Vmax)  # Clamp between supported values
      # Compute L2 projection of Tz onto fixed support z
      b = (Tz - self.Vmin) / self.delta_z  # b = (Tz - Vmin) / Δz
      l, u = b.floor().to(torch.int64), b.ceil().to(torch.int64)
      # Fix disappearing probability mass when l = b = u (b is int)
      l[(u > 0) * (l == u)] -= 1
      u[(l < (self.atoms - 1)) * (l == u)] += 1

      # Distribute probability of Tz
      m = states.new_zeros(self.batch_size, self.atoms)
      offset = torch.linspace(0, ((self.batch_size - 1) * self.atoms), self.batch_size).unsqueeze(1).expand(self.batch_size, self.atoms).to(actions)
      m.view(-1).index_add_(0, (l + offset).view(-1), (pns_a * (u.float() - b)).view(-1))  # m_l = m_l + p(s_t+n, a*)(u - b)
      m.view(-1).index_add_(0, (u + offset).view(-1), (pns_a * (b - l.float())).view(-1))  # m_u = m_u + p(s_t+n, a*)(b - l)

    loss = -torch.sum(m * log_ps_a, 1)  # Cross-entropy loss (minimises DKL(m||p(s_t, a_t)))
    self.online_net.zero_grad()
    (weights * loss).mean().backward()  # Backpropagate importance-weighted minibatch loss
    clip_grad_norm_(self.online_net.parameters(), self.norm_clip)  # Clip gradients by L2 norm
    self.optimiser.step()

    mem.update_priorities(idxs, loss.detach().cpu().numpy())  # Update priorities of sampled transitions

  def update_target_net(self):
    self.target_net.load_state_dict(self.online_net.state_dict())
    
  def train(self):
    self.online_net.train()

  def eval(self):
    self.online_net.eval()

