import signal
import time
import os
from functools import wraps, partial
import errno
import math
import random
import numpy as np

class TimeoutError(BaseException):
    pass

def timeout(seconds=10, error_message=os.strerror(errno.ETIME)):

    def decorator(func):

        def _handle_timeout(repeat_id, signum, frame):
            signal.signal(signal.SIGALRM, partial(_handle_timeout, repeat_id + 1))
            signal.alarm(seconds)
            raise TimeoutError(error_message)

        def wrapper(*args, **kwargs):
            old_signal = signal.signal(signal.SIGALRM, partial(_handle_timeout, 0))
            old_time_left = signal.alarm(seconds)
            if 0 < old_time_left < seconds:  # do not exceed previous timer
                signal.alarm(old_time_left)
            start_time = time.time()
            try:
                result = func(*args, **kwargs)
            finally:
                elapsed_time = time.time() - start_time
                signal.signal(signal.SIGALRM, old_signal)
                if old_time_left > 0:
                    remaining_time = max(0, math.ceil(old_time_left - elapsed_time))
                    signal.alarm(remaining_time)
                else:
                    signal.alarm(0)
            return result

        return wraps(func)(wrapper)

    return decorator

if __name__ == '__main__':
    # Usage example
    @timeout(5)  # Timeout after 5 seconds
    def my_function(param):
        # Simulate a cpu-intensive task
        x = np.random.randint(0,10,size = 10**9)
        return "Function Completed"

    try:
        print(my_function(1))
    except TimeoutError as e:
        print(e)


