"""
This training script can be run both on a single gpu in debug mode,
and also in a larger training run with distributed data parallel (ddp).

To run on a single GPU, example:
$ python train.py --batch_size=32 --compile=False

To run with DDP on 4 gpus on 1 node, example:
$ torchrun --standalone --nproc_per_node=4 train.py

To run with DDP on 4 gpus across 2 nodes, example:
- Run on the first (master) node with example IP 123.456.123.456:
$ torchrun --nproc_per_node=8 --nnodes=2 --node_rank=0 --master_addr=123.456.123.456 --master_port=1234 train.py
- Run on the worker node:
$ torchrun --nproc_per_node=8 --nnodes=2 --node_rank=1 --master_addr=123.456.123.456 --master_port=1234 train.py
(If your cluster does not have Infiniband interconnect prepend NCCL_IB_DISABLE=1)
"""

import os
import time
import math
import pickle
from contextlib import nullcontext
import numpy as np
import io
import numpy as np
import torch

from torch.nn.parallel import DistributedDataParallel as DDP
from torch.distributed import init_process_group, destroy_process_group
from torch.nn import functional as F

from alpha_integrate.train.model import GPTConfig, GPT
from alpha_integrate.synthetic_data.tokenization import tokenize, WORD2ID, PAD, detokenize

def main(args):
    
    dataset = args.dataset
    model_name = args.model_name

    # -----------------------------------------------------------------------------
    # default config values designed to train a gpt2 (124M) on OpenWebText
    # I/O
    out_dir = 'alpha_integrate/train/out-integrate'
    eval_interval = 250 
    eval_iters = 100
    log_interval = 50 # don't print too too often
    eval_only = False # if True, script exits right after the first eval
    always_save_checkpoint = False # if True, always save a checkpoint after each eval
    init_from = 'resume' # 'scratch' or 'resume' or 'gpt2*'

    # wandb logging
    wandb_log = False # disabled by default
    wandb_project = 'integrate'
    wandb_run_name = 'integrate' # 'run' + str(time.time())

    # dataset
    gradient_accumulation_steps = 1
    batch_size = 256
    block_size = 1024 # context of up to 1024 previous characters
    # model
    n_layer = 6
    n_head = 6
    n_embd = 384
    dropout = 0.2
    bias = False # do we use bias inside LayerNorm and Linear layers?
    # adamw optimizer
    learning_rate = 1e-3 # with baby networks can afford to go a bit higher
    max_iters = 50000
    lr_decay_iters = 50000 # make equal to max_iters usually
    min_lr = 1e-4 # learning_rate / 10 usually
    beta2 = 0.99 # make a bit larger because number of tokens per iter is small
    beta1 = 0.9
    grad_clip = 1.0 # clip gradients at this value, or disable if == 0.0

    # learning rate decay settings
    decay_lr = True # whether to decay the learning rate

    warmup_iters = 100 # not super necessary potentially
    weight_decay = 0.1 
    # DDP settings
    backend = 'nccl' # 'nccl', 'gloo', etc.
    # system
    device = 'cuda' # examples: 'cpu', 'cuda', 'cuda:0', 'cuda:1' etc., or try 'mps' on macbooks
    dtype = 'bfloat16' if torch.cuda.is_available() and torch.cuda.is_bf16_supported() else 'float16' # 'float32', 'bfloat16', or 'float16', the latter will auto implement a GradScaler
    compile = True # use PyTorch 2.0 to compile the model to be faster
    # -----------------------------------------------------------------------------
    config_keys = [k for k,v in globals().items() if not k.startswith('_') and isinstance(v, (int, float, bool, str))]
    exec(open('alpha_integrate/train/configurator.py').read()) # overrides from command line or config file
    config = {k: globals()[k] for k in config_keys} # will be useful for logging
    # -----------------------------------------------------------------------------
    print(config)

    # various inits, derived attributes, I/O setup
    ddp = int(os.environ.get('RANK', -1)) != -1 # is this a ddp run?
    if ddp:
        init_process_group(backend=backend)
        ddp_rank = int(os.environ['RANK'])
        ddp_local_rank = int(os.environ['LOCAL_RANK'])
        ddp_world_size = int(os.environ['WORLD_SIZE'])
        device = f'cuda:{ddp_local_rank}'
        torch.cuda.set_device(device)
        master_process = ddp_rank == 0 # this process will do logging, checkpointing etc.
        seed_offset = ddp_rank # each process gets a different seed
        # world_size number of processes will be training simultaneously, so we can scale
        # down the desired gradient accumulation iterations per process proportionally
        assert gradient_accumulation_steps % ddp_world_size == 0
        gradient_accumulation_steps //= ddp_world_size
    else:
        # if not ddp, we are running on a single gpu, and one process
        master_process = True
        seed_offset = 0
        ddp_world_size = 1
    tokens_per_iter = gradient_accumulation_steps * ddp_world_size * batch_size * block_size
    print(f"tokens per iteration will be: {tokens_per_iter:,}")

    if master_process:
        os.makedirs(out_dir, exist_ok=True)
    torch.manual_seed(1337 + seed_offset)
    torch.backends.cuda.matmul.allow_tf32 = True # allow tf32 on matmul
    torch.backends.cudnn.allow_tf32 = True # allow tf32 on cudnn
    device_type = 'cuda' if 'cuda' in device else 'cpu' # for later use in torch.autocast
    # note: float16 data type will automatically use a GradScaler
    ptdtype = {'float32': torch.float32, 'bfloat16': torch.bfloat16, 'float16': torch.float16}[dtype]
    ctx = nullcontext() if device_type == 'cpu' else torch.amp.autocast(device_type=device_type, dtype=ptdtype)

    # create a dataloader

    path_train = f'alpha_integrate/synthetic_data/final_steps_dataset/{dataset}/train.txt'
    with io.open(path_train, mode='r', encoding='utf-8') as f:
        lines_train = [line for line in f]

    path_val = f'alpha_integrate/synthetic_data/final_steps_dataset/{dataset}/val.txt'
    with io.open(path_val, mode='r', encoding='utf-8') as f:
        lines_val = [line for line in f]

    PAD_TOKEN = WORD2ID[PAD]
    VOCAB_SIZE = len(WORD2ID)

    def get_sample(block_size, split):

        if split == 'train':
            lines = lines_train
        elif split == 'val':
            lines = lines_val
        else:
            raise ValueError(f"Invalid split")

        idx = np.random.randint(0, len(lines))
        line = lines[idx]
        tokenized, start_id, end_id = tokenize(line) 
        if len(tokenized) > block_size - 1:
            return get_sample(block_size, split)
        tokenized = tokenized + [WORD2ID[PAD]] * (block_size - len(tokenized))
        #print(tokenized)
        return np.array(tokenized), start_id, end_id

    def get_batch(block_size, batch_size, device_type, device, split):

        lines = []
        start_idx = [0]*batch_size
        end_idx = [0]*batch_size

        for i in range(batch_size):
            line, start_id, end_id = get_sample(block_size, split)
            lines.append(line)
            start_idx[i] = start_id
            end_idx[i] = end_id
        
        x = torch.stack([torch.from_numpy(line.astype(np.int64)) for line in lines])
        y = torch.stack([torch.from_numpy(line.astype(np.int64)) for line in lines])
        if device_type == 'cuda':
            # pin arrays x,y, which allows us to move them to GPU asynchronously (non_blocking=True)
            x, y = x.pin_memory().to(device, non_blocking=True), y.pin_memory().to(device, non_blocking=True)
        else:
            x, y = x.to(device), y.to(device)


        return x, y, start_idx, end_idx

    # init these up here, can override if init_from='resume' (i.e. from a checkpoint)
    iter_num = 0
    best_val_loss = 1e9

    meta_vocab_size = VOCAB_SIZE
    print(f"found vocab_size = {meta_vocab_size}")

    # model init
    model_args = dict(n_layer=n_layer, n_head=n_head, n_embd=n_embd, block_size=block_size,
                    bias=bias, vocab_size=None, dropout=dropout) # start with model_args from command line
    if init_from == 'scratch':
        # init a new model from scratch
        print("Initializing a new model from scratch")
        # determine the vocab size we'll use for from-scratch training
        if meta_vocab_size is None:
            print("defaulting to vocab_size of GPT-2 to 50304 (50257 rounded up for efficiency)")
        model_args['vocab_size'] = meta_vocab_size if meta_vocab_size is not None else 50304
        gptconf = GPTConfig(**model_args)
        model = GPT(gptconf)
    elif init_from == 'resume':
        print(f"Resuming training from {out_dir}")
        # resume training from a checkpoint.
        ckpt_path = os.path.join(out_dir, f'{model_name}.pt')
        checkpoint = torch.load(ckpt_path, map_location=device)
        checkpoint_model_args = checkpoint['model_args']
        # force these config attributes to be equal otherwise we can't even resume training
        # the rest of the attributes (e.g. dropout) can stay as desired from command line
        for k in ['n_layer', 'n_head', 'n_embd', 'block_size', 'bias', 'vocab_size']:
            model_args[k] = checkpoint_model_args[k]
        # create the model
        gptconf = GPTConfig(**model_args)
        model = GPT(gptconf)
        state_dict = checkpoint['model']
        # fix the keys of the state dictionary :(
        # honestly no idea how checkpoints sometimes get this prefix, have to debug more
        unwanted_prefix = '_orig_mod.'
        for k,v in list(state_dict.items()):
            if k.startswith(unwanted_prefix):
                state_dict[k[len(unwanted_prefix):]] = state_dict.pop(k)
        model.load_state_dict(state_dict)
        iter_num = checkpoint['iter_num']
        best_val_loss = checkpoint['best_val_loss']
    elif init_from.startswith('gpt2'):
        print(f"Initializing from OpenAI GPT-2 weights: {init_from}")
        # initialize from OpenAI GPT-2 weights
        override_args = dict(dropout=dropout)
        model = GPT.from_pretrained(init_from, override_args)
        # read off the created config params, so we can store them into checkpoint correctly
        for k in ['n_layer', 'n_head', 'n_embd', 'block_size', 'bias', 'vocab_size']:
            model_args[k] = getattr(model.config, k)
    # crop down the model block size if desired, using model surgery
    if block_size < model.config.block_size:
        model.crop_block_size(block_size)
        model_args['block_size'] = block_size # so that the checkpoint will have the right value
    model.to(device)

    # initialize a GradScaler. If enabled=False scaler is a no-op
    scaler = torch.cuda.amp.GradScaler(enabled=(dtype == 'float16'))

    # optimizer
    optimizer = model.configure_optimizers(weight_decay, learning_rate, (beta1, beta2), device_type)
    if init_from == 'resume':
        optimizer.load_state_dict(checkpoint['optimizer'])
    checkpoint = None # free up memory

    # compile the model
    if compile:
        print("compiling the model... (takes a ~minute)")
        unoptimized_model = model
        model = torch.compile(model) # requires PyTorch 2.0

    # wrap model into DDP container
    if ddp:
        model = DDP(model, device_ids=[ddp_local_rank])

    def get_loss(X, Y, start_idx, end_idx):
        with ctx:
            logits, _ = model(X, Y)

            # apply cross entropy loss for each sample between start_idx and end_idx (different for every sample)
            loss_list = []
            weights = torch.zeros(batch_size).to(device)
            for i in range(batch_size):
                flat_logits = logits[i, start_idx[i]:end_idx[i]].view(-1, logits.size(-1))
                target = Y[i, start_idx[i]+1:end_idx[i]+1].view(-1)
                if flat_logits.shape[0] != target.shape[0]:
                    #print(f"flat_logits: {flat_logits.shape}")
                    #print(f"target: {target.shape}")
                    #print(f"start_idx: {start_idx[i]}")
                    #print(f"end_idx: {end_idx[i]}")
                    pass

                loss_list.append(F.cross_entropy(flat_logits, target, ignore_index = PAD_TOKEN))
                weights[i] = end_idx[i] - start_idx[i]
            
            # loss is weighted mean of losses
            loss = torch.mean(torch.stack(loss_list) * weights / torch.sum(weights))

        return loss


    # helps estimate an arbitrarily accurate loss over either split using many batches
    @torch.no_grad()
    def estimate_loss():
        out = {}
        model.eval()
        for split in ['train', 'val']:
            losses = torch.zeros(eval_iters)
            for k in range(eval_iters):
                X, Y, start_idx, end_idx = get_batch(block_size, batch_size*2, device_type, device, split)
                loss = get_loss(X, Y, start_idx, end_idx)
                #print(f"Loss: {loss}")
                losses[k] = loss.item()
            out[split] = losses.mean()
        model.train()
        return out

    # learning rate decay scheduler (cosine with warmup)
    def get_lr(it):
        # 1) linear warmup for warmup_iters steps
        if it < warmup_iters:
            return learning_rate * it / warmup_iters
        # 2) if it > lr_decay_iters, return min learning rate
        if it > lr_decay_iters:
            return min_lr
        # 3) in between, use cosine decay down to min learning rate
        decay_ratio = (it - warmup_iters) / (lr_decay_iters - warmup_iters)
        assert 0 <= decay_ratio <= 1
        coeff = 0.5 * (1.0 + math.cos(math.pi * decay_ratio)) # coeff ranges 0..1
        return min_lr + coeff * (learning_rate - min_lr)

    # logging
    if wandb_log and master_process:
        import wandb
        wandb.init(project=wandb_project, name=wandb_run_name, config=config)

    # training loop
    X, Y, start_idx, end_idx = get_batch(block_size, batch_size, device_type, device, 'train') # fetch the very first batch
    #print(f"X: {X}")
    #print(f"Y: {Y}")
    #print(f"start_idx: {start_idx}")
    #print(f"end_idx: {end_idx}")

    t0 = time.time()
    local_iter_num = 0 # number of iterations in the lifetime of this process
    raw_model = model.module if ddp else model # unwrap DDP container if needed
    running_mfu = -1.0
    while True:

        # determine and set the learning rate for this iteration
        lr = get_lr(iter_num) if decay_lr else learning_rate
        for param_group in optimizer.param_groups:
            param_group['lr'] = lr

        # evaluate the loss on train/val sets and write checkpoints
        if iter_num % eval_interval == 0 and master_process:
            losses = estimate_loss()
            print(f"step {iter_num}: train loss {losses['train']:.4f}, val loss {losses['val']:.4f}")
            if wandb_log:
                wandb.log({
                    "iter": iter_num,
                    "train/loss": losses['train'],
                    "val/loss": losses['val'],
                    "lr": lr,
                    "mfu": running_mfu*100, # convert to percentage
                })
            if losses['val'] < best_val_loss or always_save_checkpoint:
                best_val_loss = losses['val']
                if iter_num > 0:
                    checkpoint = {
                        'model': raw_model.state_dict(),
                        'optimizer': optimizer.state_dict(),
                        'model_args': model_args,
                        'iter_num': iter_num,
                        'best_val_loss': best_val_loss,
                        'config': config,
                    }
                    print(f"saving checkpoint to {out_dir}")
                    torch.save(checkpoint, os.path.join(out_dir, f'{model_name  }.pt'))
        if iter_num == 0 and eval_only:
            break

        # forward backward update, with optional gradient accumulation to simulate larger batch size
        # and using the GradScaler if data type is float16
        for micro_step in range(gradient_accumulation_steps):
            if ddp:
                # in DDP training we only need to sync gradients at the last micro step.
                # the official way to do this is with model.no_sync() context manager, but
                # I really dislike that this bloats the code and forces us to repeat code
                # looking at the source of that context manager, it just toggles this variable
                model.require_backward_grad_sync = (micro_step == gradient_accumulation_steps - 1)
            with ctx:
                loss = get_loss(X, Y, start_idx, end_idx)
                loss = loss / gradient_accumulation_steps # scale the loss to account for gradient accumulation
            # immediately async prefetch next batch while model is doing the forward pass on the GPU
            X, Y, start_idx, end_idx = get_batch(block_size, batch_size, device_type, device, 'train')
            
            # backward pass, with gradient scaling if training in fp16
            scaler.scale(loss).backward()
        # clip the gradient
        if grad_clip != 0.0:
            scaler.unscale_(optimizer)
            torch.nn.utils.clip_grad_norm_(model.parameters(), grad_clip)
        # step the optimizer and scaler if training in fp16
        scaler.step(optimizer)
        scaler.update()
        # flush the gradients as soon as we can, no need for this memory anymore
        optimizer.zero_grad(set_to_none=True)

        # timing and logging
        t1 = time.time()
        dt = t1 - t0
        t0 = t1
        if iter_num % log_interval == 0 and master_process:
            # get loss as float. note: this is a CPU-GPU sync point
            # scale up to undo the division above, approximating the true total loss (exact would have been a sum)
            lossf = loss.item() * gradient_accumulation_steps
            if local_iter_num >= 5: # let the training loop settle a bit
                mfu = raw_model.estimate_mfu(batch_size * gradient_accumulation_steps, dt)
                running_mfu = mfu if running_mfu == -1.0 else 0.9*running_mfu + 0.1*mfu
            print(f"iter {iter_num}: loss {lossf:.4f}, time {dt*1000:.2f}ms, mfu {running_mfu*100:.2f}%")
        iter_num += 1
        local_iter_num += 1

        # termination conditions
        if iter_num > max_iters:
            break

    if ddp:
        destroy_process_group()

if __name__ == '__main__':
    import argparse
    parser = argparse.ArgumentParser(description='Train an integrator model')
    parser.add_argument('config', type=str, nargs='?', default='alpha_integrate/train/config/train_integrate.py', help='Path to the configuration file (optional)')
    parser.add_argument('--dataset', type=str, required=True, help='Name of the dataset to train on')
    parser.add_argument('--model_name', type=str, required=True, help='Name of the model to train')
    args = parser.parse_args()
    main(args)